package ru.yandex.crm.apphost.kotlin.common.http.parser

import NAppHostHttp.Http.THttpRequest
import NAppHostHttp.Http.THttpRequest.EMethod
import NAppHostTvmUserTicket.TvmUserTicket.TTvmUserTicket
import mu.KotlinLogging
import ru.yandex.crm.apphost.kotlin.common.apphost.AppHostPathHandler
import ru.yandex.crm.apphost.kotlin.common.apphost.PROTO_HTTP_REQUEST
import ru.yandex.crm.apphost.kotlin.common.apphost.USER_TICKET
import ru.yandex.crm.apphost.kotlin.common.extensions.getAppHostType
import ru.yandex.crm.apphost.kotlin.common.extensions.getHeader
import ru.yandex.crm.apphost.kotlin.common.http.SERVICE_TICKET_HEADER_NAME
import ru.yandex.crm.apphost.kotlin.common.http.USER_TICKET_HEADER_NAME
import ru.yandex.crm.apphost.proto.userselector.UserSelector.YandexService
import ru.yandex.crm.apphost.proto.userselector.UserSelector.YandexUser
import ru.yandex.passport.tvmauth.TvmClient
import ru.yandex.web.apphost.api.request.RequestContext

abstract class HttpParserHandlerBase(private val tvmClient: TvmClient) : AppHostPathHandler {
    companion object {
        private val logger = KotlinLogging.logger { }
    }

    private val routes = mutableMapOf<Route, RouteContext.() -> Unit>()

    fun route(
        method: EMethod,
        maskString: String,
        action: RouteContext.() -> Unit
    ) {
        val routeString = if (maskString.startsWith("/")) maskString.drop(1) else maskString
        routes[Route(method, routeString)] = action
    }

    fun get(
        maskString: String,
        action: RouteContext.() -> Unit
    ) {
        route(EMethod.Get, maskString, action)
    }

    fun post(
        maskString: String,
        action: RouteContext.() -> Unit
    ) {
        route(EMethod.Post, maskString, action)
    }

    fun put(
        maskString: String,
        action: RouteContext.() -> Unit
    ) {
        route(EMethod.Put, maskString, action)
    }

    fun patch(
        maskString: String,
        action: RouteContext.() -> Unit
    ) {
        route(EMethod.Patch, maskString, action)
    }

    fun delete(
        maskString: String,
        action: RouteContext.() -> Unit
    ) {
        route(EMethod.Delete, maskString, action)
    }

    override fun handle(ctx: RequestContext) {
        val request = ctx.getSingleRequestItem(PROTO_HTTP_REQUEST)
            .getProtobufData(THttpRequest.getDefaultInstance())

        handleRequest(request, ctx)
    }

    fun handleRequest(request: THttpRequest, ctx: RequestContext) {
        val parseResult = routes
            .map { it.key.parse(request, ctx) }
            .filterNotNull()
            .maxByOrNull { it.strength }

        if (parseResult != null) {
            logger.info("Route to ${parseResult.route.routeString}")
            val routeAction = routes[parseResult.route]

            routeAction?.run {
                invoke(parseResult.routeContext)
                parseTvmTickets(request, ctx)
                logger.info("Route to ${parseResult.route.routeString} handled successfully")
            } ?: logger.info("Route to ${parseResult.route.routeString} not handled")
        }
    }

    private fun parseTvmTickets(request: THttpRequest, context: RequestContext) {
        with(request.getHeader(USER_TICKET_HEADER_NAME)) {
            if (this === null) {
                return@with
            }

            val checkedUserTicket = tvmClient.checkUserTicket(this.value)
            if (checkedUserTicket.booleanValue()) {
                context.addProtobufItem(
                    YandexUser.getDescriptor().getAppHostType(),
                    YandexUser.newBuilder().setUId(checkedUserTicket.defaultUid).build()
                )
                context.addProtobufItem(
                    USER_TICKET,
                    TTvmUserTicket.newBuilder().setUserTicket(this.value).build()
                )
                logger.info { "Authenticated with user ticket" }
            } else {
                logger.debug { "Unable to check user ticket: ${checkedUserTicket.debugInfo()}" }
            }
        }
        with(request.getHeader(SERVICE_TICKET_HEADER_NAME)) {
            if (this === null) {
                return@with
            }

            val checkedServiceTicket = tvmClient.checkServiceTicket(this.value)
            if (checkedServiceTicket.booleanValue()) {
                context.addProtobufItem(
                    YandexService.getDescriptor().getAppHostType(),
                    YandexService.newBuilder().setTvmId(checkedServiceTicket.src).build()
                )
                logger.info { "Authenticated with service ticket" }
                if (checkedServiceTicket.issuerUid != 0L) {
                    logger.warn { "Service ticket issued by SSH user ${checkedServiceTicket.issuerUid}" }
                }
            } else {
                logger.debug { "Unable to check service ticket: ${checkedServiceTicket.debugInfo()}" }
            }
        }
    }
}
