package ru.yandex.crm.apphost.kotlin.common.http.parser

import NAppHostHttp.Http
import ru.yandex.web.apphost.api.request.RequestContext
import java.net.URLDecoder
import java.nio.charset.Charset

class Route internal constructor(
    internal val method: Http.THttpRequest.EMethod,
    internal val routeString: String
) {
    enum class PathTokenType {
        CONSTANT,
        PARAMETER,
        ANY;
    }

    data class PathToken(val type: PathTokenType, val tokenValue: String = "")

    data class RequestParam(val name: String, val alias: String)

    data class ParseResult(val route: Route, val routeContext: RouteContext, val strength: Int)

    private val pathTokens: List<PathToken>
    private val requestParams: List<RequestParam>

    init {
        val splitString = routeString.split('?')

        pathTokens = parsePath(splitString.getOrElse(0) { "" })
        requestParams = parseRequestParams(splitString.getOrElse(1) { "" })
    }

    internal fun parse(request: Http.THttpRequest, context: RequestContext): ParseResult? {
        if (method != request.method)
            return null

        val splitString = request.path.split('?')
        val (pathParams, strength) = collectPathParams(splitString.getOrElse(0) { "" })
            ?: return null
        val requestParams = collectRequestParams(splitString.getOrElse(1) { "" })
        val routeContext = RouteContext(pathParams, requestParams, context, request)

        return ParseResult(this, routeContext, strength)
    }

    private fun collectPathParams(currentRequestPath: String): Pair<Map<String, String>, Int>? {
        if (pathTokens.isEmpty()) {
            return emptyMap<String, String>() to 0
        }
        val resultParams = mutableMapOf<String, String>()
        val pathParts = currentRequestPath.split("/").filter { it.isNotBlank() }

        var strength = 0
        pathParts.forEachIndexed { index, part ->
            if (index >= pathTokens.size) {
                return null
            }

            if (pathTokens[index].type == PathTokenType.CONSTANT) {
                if (pathTokens[index].tokenValue != part) {
                    return null
                } else {
                    strength++
                }
            }

            if (pathTokens[index].type == PathTokenType.PARAMETER) {
                resultParams[pathTokens[index].tokenValue] = part
            }
        }

        return resultParams to strength
    }

    private fun collectRequestParams(currentRequestParams: String): Map<String, String> {
        if (requestParams.isEmpty()) {
            return emptyMap()
        }
        val resultParams = mutableMapOf<String, String>()
        val requestParts = currentRequestParams.split('&').filter { it.isNotBlank() }

        requestParts.forEach { part ->
            val (paramName, paramValue) = part.split('=')
            val paramAlias = requestParams.find { it.name == paramName }?.alias
                ?: return@forEach
            resultParams[paramAlias] = URLDecoder.decode(paramValue, "UTF-8")
        }

        return resultParams
    }

    private fun parsePath(path: String): List<PathToken> {
        if (path == "*") {
            return emptyList()
        }
        val result = mutableListOf<PathToken>()
        val tokens = path.split('/')

        tokens.forEach { rawToken ->
            if (rawToken.startsWith("{") && rawToken.endsWith("}")) {
                result.add(PathToken(PathTokenType.PARAMETER, rawToken.trim('{', '}')))
            } else if (rawToken == "*") {
                result.add(PathToken(PathTokenType.ANY))
            } else {
                result.add(PathToken(PathTokenType.CONSTANT, rawToken))
            }
        }

        return result
    }

    private fun parseRequestParams(requestParams: String): List<RequestParam> {
        val result = mutableListOf<RequestParam>()
        val params = requestParams.split('&')

        params.forEach { rawParam ->
            val paramData = rawParam.split('=')
            val paramName = paramData.getOrNull(0)
                ?: return@forEach
            val paramAlias = paramData.getOrNull(1)?.trim('{', '}')
                ?: return@forEach

            result.add(RequestParam(paramName, paramAlias))
        }

        return result
    }
}
