package ru.yandex.crm.apphost.kotlin.common.ut.service

import NAppHostHttp.Http.THttpRequest
import org.junit.jupiter.api.Assertions.assertEquals
import org.junit.jupiter.api.Assertions.assertNull
import org.junit.jupiter.api.Test
import ru.yandex.crm.apphost.kotlin.common.extensions.getAppHostType
import ru.yandex.crm.apphost.kotlin.common.http.parser.HttpParserHandlerBase
import ru.yandex.crm.apphost.proto.departmentmanager.Departmentmanager
import ru.yandex.passport.tvmauth.*
import ru.yandex.passport.tvmauth.roles.Roles
import ru.yandex.web.apphost.api.request.RequestContextBase
import ru.yandex.web.apphost.grpc.proto.TServiceRequest

open class HttpParserHandlerBaseTests {

    class TestHttpParserHandler : HttpParserHandlerBase(tvmClient = MockTvmClient()) {
        override val path: String = "test"
    }

    var ctx: RequestContextBase = RequestContextBase(TServiceRequest.getDefaultInstance())

    @Test
    fun `get route with constant path matches`() {
        // Arrange
        val handler = TestHttpParserHandler()
        var executed = false
        handler.get("test1/test2") {
            executed = true
        }

        val request = THttpRequest.newBuilder()
            .setPath("/test1/test2")
            .setMethod(THttpRequest.EMethod.Get)
            .build()

        // Act
        handler.handleRequest(request, ctx)

        // Assert
        assertEquals(true, executed)
    }

    @Test
    fun `get route with constant path not matches`() {
        // Arrange
        val handler = TestHttpParserHandler()
        var executed = false
        handler.get("test1/test2") {
            executed = true
        }

        val request = THttpRequest.newBuilder()
            .setPath("/test1/test3")
            .setMethod(THttpRequest.EMethod.Get)
            .build()

        // Act
        handler.handleRequest(request, ctx)

        // Assert
        assertEquals(false, executed)
    }

    @Test
    fun `get route with asterisk full path matches`() {
        // Arrange
        val handler = TestHttpParserHandler()
        var executed = false
        handler.get("*") {
            executed = true
        }

        val request = THttpRequest.newBuilder()
            .setPath("/test1/test3")
            .setMethod(THttpRequest.EMethod.Get)
            .build()

        // Act
        handler.handleRequest(request, ctx)

        // Assert
        assertEquals(true, executed)
    }

    @Test
    fun `get route with asterisk and parameter matches`() {
        // Arrange
        val handler = TestHttpParserHandler()
        var executed = false
        handler.get("*/{test1}") {
            executed = true
            assertEquals("123", pathParameters["test1"])
        }

        val request = THttpRequest.newBuilder()
            .setPath("/test/123")
            .setMethod(THttpRequest.EMethod.Get)
            .build()

        // Act
        handler.handleRequest(request, ctx)

        // Assert
        assertEquals(true, executed)
    }

    @Test
    fun `post route with parameter matches`() {
        // Arrange
        val handler = TestHttpParserHandler()
        var executed = false
        handler.post("test1/{test2}/{test3}") {
            executed = true
            assertEquals("123", pathParameters["test2"])
            assertEquals("456", pathParameters["test3"])
        }

        val request = THttpRequest.newBuilder()
            .setPath("/test1/123/456")
            .setMethod(THttpRequest.EMethod.Post)
            .build()

        // Act
        handler.handleRequest(request, ctx)

        // Assert
        assertEquals(true, executed)
    }

    @Test
    fun `get route with parameter and constant matches constant`() {
        // Arrange
        val handler = TestHttpParserHandler()
        var executed = false
        handler.get("test1/{test2}") {
        }

        handler.get("test1/list") {
            executed = true
        }

        val request = THttpRequest.newBuilder()
            .setPath("/test1/list")
            .setMethod(THttpRequest.EMethod.Get)
            .build()

        // Act
        handler.handleRequest(request, ctx)

        // Assert
        assertEquals(true, executed)
    }

    @Test
    fun `put route with one request parameter`() {
        val handler = TestHttpParserHandler()
        var executed = false
        handler.put("test1/test2/test3?param1={param2}") {
            executed = true
            assertEquals("123", requestParameters["param2"])
        }

        val request = THttpRequest.newBuilder()
            .setPath("/test1/test2/test3?param1=123")
            .setMethod(THttpRequest.EMethod.Put)
            .build()

        handler.handleRequest(request, ctx)

        assertEquals(true, executed)
    }

    @Test
    fun `delete route with no request parameters`() {
        val handler = TestHttpParserHandler()
        var executed = false
        handler.delete("test1/test2/test3") {
            executed = true
            assertNull(requestParameters["param1"])
        }

        val request = THttpRequest.newBuilder()
            .setPath("/test1/test2/test3?param1=123")
            .setMethod(THttpRequest.EMethod.Delete)
            .build()

        handler.handleRequest(request, ctx)

        assertEquals(true, executed)
    }

    @Test
    fun `post route with request params but they not passed`() {
        val handler = TestHttpParserHandler()
        var executed = false
        handler.post("test1/test2/test3?param2={param1}") {
            executed = true
            assertNull(requestParameters["param1"])
        }

        val request = THttpRequest.newBuilder()
            .setPath("/test1/test2/test3")
            .setMethod(THttpRequest.EMethod.Post)
            .build()

        handler.handleRequest(request, ctx)

        assertEquals(true, executed)
    }

    @Test
    fun `get route with many request parameters`() {
        val handler = TestHttpParserHandler()
        var executed = false
        handler.post("test1/test2/test3?param1={param1}&param2={param2}&param3={param3}") {
            executed = true
            assertEquals("1", requestParameters["param1"])
            assertEquals("2", requestParameters["param2"])
            assertEquals("3", requestParameters["param3"])
        }

        val request = THttpRequest.newBuilder()
            .setPath("/test1/test2/test3?param3=3&param2=2&param1=1")
            .setMethod(THttpRequest.EMethod.Post)
            .build()

        handler.handleRequest(request, ctx)

        assertEquals(true, executed)
    }

    @Test
    fun `complicated post route with many path and request parameters`() {
        // Arrange
        val handler = TestHttpParserHandler()
        var executed = false
        handler.post("test1/{test2}/{test3}/test4?param1={test5}&param6={test123}") {
            executed = true
            assertEquals("123", pathParameters["test2"])
            assertEquals("456", pathParameters["test3"])
            assertEquals("1", requestParameters["test5"])
            assertEquals("2", requestParameters["test123"])
            assertNull(requestParameters["param1"])
        }

        val request = THttpRequest.newBuilder()
            .setPath("/test1/123/456?param1=1&param6=2&notparam=2")
            .setMethod(THttpRequest.EMethod.Post)
            .build()

        // Act
        handler.handleRequest(request, ctx)

        // Assert
        assertEquals(true, executed)
    }

    @Test
    fun `get department users route handles correct`() {
        // Arrange
        val handler = TestHttpParserHandler()
        var executed = false
        handler.get("department/{organizationId}/{departmentId}/users") {
            executed = true
            assertEquals("5", pathParameters["organizationId"])
            assertEquals("d3285355-e708-40ab-81cc-5d1b0dbfb1ce", pathParameters["departmentId"])
        }

        handler.get("department/{organizationId}/{departmentId}/responsible?includeParentDepartments={includeParent}") {
        }

        val request = THttpRequest.newBuilder()
            .setPath("/department/5/d3285355-e708-40ab-81cc-5d1b0dbfb1ce/users")
            .setMethod(THttpRequest.EMethod.Get)
            .build()

        // Act
        handler.handleRequest(request, ctx)

        // Assert
        assertEquals(true, executed)
    }

    class MockTvmClient : TvmClient {
        override fun getStatus(): ClientStatus {
            return STATUS
        }

        override fun getServiceTicketFor(alias: String): String {
            return "xxx-ticket"
        }

        override fun getServiceTicketFor(tvmId: Int): String {
            return "xxx-ticket"
        }

        override fun checkServiceTicket(ticketBody: String): CheckedServiceTicket {
            return Unittest.createServiceTicket(TicketStatus.OK, 1)
        }

        override fun checkUserTicket(ticket: String): CheckedUserTicket {
            return if (ticket.startsWith(TVM_USER_TICKET_PREFIX)) {
                Unittest.createUserTicket(
                    TicketStatus.OK,
                    ticket.substring(TVM_USER_TICKET_PREFIX.length).toLong(),
                    arrayOfNulls(0),
                    LongArray(0)
                )
            } else {
                Unittest.createUserTicket(TicketStatus.EXPIRED, 1, arrayOfNulls(0), LongArray(0))
            }
        }

        override fun checkUserTicket(ticketBody: String, overridedBbEnv: BlackboxEnv): CheckedUserTicket {
            return checkUserTicket(ticketBody)
        }

        override fun getRoles(): Roles =
            throw UnsupportedOperationException("Not implemented")

        override fun close() {}

        companion object {
            private val STATUS = ClientStatus(ClientStatus.Code.OK, "")
            const val TVM_USER_TICKET_PREFIX = "TVM-USER-"
        }
    }
}
