package ru.yandex.crm.apphost.kotlin.common.ut.validator

import org.junit.jupiter.api.Assertions.assertTrue
import org.junit.jupiter.api.Test
import ru.yandex.crm.apphost.kotlin.common.validator.ModelValidator
import ru.yandex.crm.apphost.kotlin.common.validator.emptyValidator
import ru.yandex.crm.proto.gallifrey.departmentmanager.Departmentmanager.DepartmentData
import ru.yandex.crm.proto.gallifrey.departmentmanager.Departmentmanager.DepartmentName
import ru.yandex.crm.proto.gallifrey.usermanager.Usermanager.UserData

class ProtobufValidatorTests {

    @Test
    fun `test empty validator`() {
        val validator = object: ModelValidator<DepartmentData>() {
            override val configuration = emptyValidator<DepartmentData>()
        }

        val instance = DepartmentData.newBuilder()
            .setOrganizationId(1)
            .setParentDepartmentId("123")
            .build()

        val result = validator.validate(instance)

        assertTrue(result.isEmpty())
    }

    @Test
    fun `test simple validator`() {
        val validator = object: ModelValidator<DepartmentData>() {
            override val configuration = emptyValidator<DepartmentData>()
                .addValidation("Test1") {
                    namesList.isEmpty()
                }
        }

        val instance = DepartmentData.newBuilder()
            .addNames(
                DepartmentName.newBuilder()
                    .setName("name")
                    .setLanguageCode("en")
                    .build()
            )
            .build()

        val result = validator.validate(instance)

        assertTrue(result.size == 1)
        assertTrue(result.first().testDescription == "Test1")
        assertTrue(result.first().checkNumber == 0)
    }

    @Test
    fun `test complicated validator with last check error`() {
        val validator = object: ModelValidator<UserData>() {
            override val configuration = emptyValidator<UserData>()
                    .addValidation("test1") {
                        firstName.isNotBlank()
                    }
                    .addValidation("test2") {
                        secondName.isNotBlank()
                    }
                    .addValidation("test3") {
                        firstName == "name"
                    }
        }

        val instance = UserData.newBuilder()
            .setFirstName("name1")
            .setSecondName("name2")
            .build()

        val result = validator.validate(instance)

        assertTrue(result.size == 1)
        assertTrue(result.first().testDescription == "test3")
        assertTrue(result.first().checkNumber == 2)
    }

    @Test
    fun `test complicated validator with first check error`() {
        val validator = object: ModelValidator<UserData>() {
            override val configuration = emptyValidator<UserData>()
                    .addValidation("test1") {
                        firstName.isNotBlank()
                    }
                    .addValidation("test2") {
                        secondName.isNotBlank()
                    }
                    .addValidation("test3") {
                        firstName == "name"
                    }
                    .addValidation("test4") {
                        firstName.isBlank()
                    }
        }

        val instance = UserData.newBuilder()
            .setSecondName("name2")
            .build()

        val result = validator.validate(instance)
        assertTrue(result.size == 2)
        assertTrue(result.first().testDescription == "test1")
        assertTrue(result.first().checkNumber == 0)
        assertTrue(result.last().testDescription == "test3")
        assertTrue(result.last().checkNumber == 2)
    }

    @Test
    fun `test complicated validator with middle check error`() {
        val validator = object: ModelValidator<UserData>() {
            override val configuration = emptyValidator<UserData>()
                    .addValidation("test1") {
                        firstName.isNotBlank()
                    }
                    .addValidation("test2") {
                        secondName.isNotBlank()
                    }
                    .addValidation("test3") {
                        firstName == "name"
                    }
                    .addValidation("test4") {
                        secondName.isBlank()
                    }
        }

        val instance = UserData.newBuilder()
            .setFirstName("name")
            .build()

        val result = validator.validate(instance)

        assertTrue(result.size == 1)
        assertTrue(result.first().testDescription == "test2")
        assertTrue(result.first().checkNumber == 1)
    }
}
