package ru.yandex.crm.apphost.kotlin.handlers.accessmanager.department

import org.jooq.Configuration
import ru.yandex.crm.apphost.kotlin.handlers.accessmanager.department.abstractions.DepartmentAccessSelector
import ru.yandex.crm.apphost.kotlin.handlers.accessmanager.department.abstractions.DepartmentAccessUpdater
import ru.yandex.crm.apphost.kotlin.handlers.accessmanager.department.abstractions.DepartmentAttributeAccess
import ru.yandex.crm.apphost.kotlin.handlers.accessmanager.department.abstractions.DepartmentEntityAccess
import ru.yandex.crm.generated.database.accessmanager.accessmanager.tables.references.DEPARTMENT_ATTRIBUTE_ACCESS
import ru.yandex.crm.generated.database.accessmanager.accessmanager.tables.references.DEPARTMENT_ENTITY_ACCESS
import java.util.UUID

class DepartmentAccessProvider(
    private val jooqConfiguration: Configuration,
) : DepartmentAccessSelector, DepartmentAccessUpdater {

    override fun getDepartmentEntitiesAccesses(
        departmentId: UUID,
        entitiesMetasIds: Collection<UUID>
    ): Collection<DepartmentEntityAccess> {
        return jooqConfiguration.dsl()
            .select(*DEPARTMENT_ENTITY_ACCESS.fields())
            .where(DEPARTMENT_ENTITY_ACCESS.DEPARTMENT_ID.eq(departmentId))
            .and(DEPARTMENT_ENTITY_ACCESS.ENTITY_META_ID.`in`(entitiesMetasIds))
            .fetchInto(DepartmentEntityAccess::class.java)
    }

    override fun updateDepartmentEntityAccess(access: DepartmentEntityAccess): DepartmentEntityAccess {
        jooqConfiguration.dsl().transaction { cfg ->
            cfg.dsl()
                .insertInto(
                    DEPARTMENT_ENTITY_ACCESS,
                    DEPARTMENT_ENTITY_ACCESS.DEPARTMENT_ID,
                    DEPARTMENT_ENTITY_ACCESS.ENTITY_META_ID,
                    DEPARTMENT_ENTITY_ACCESS.AUTHOR_ACCESS_ID,
                    DEPARTMENT_ENTITY_ACCESS.RESPONSIBLE_ACCESS_ID,
                    DEPARTMENT_ENTITY_ACCESS.FOLLOWER_ACCESS_ID,
                )
                .values(
                    access.departmentId,
                    access.entityMetaId,
                    access.access.authorAccess.value.toByte(),
                    access.access.responsibleAccess.value.toByte(),
                    access.access.followerAccess.value.toByte()
                )
                .onDuplicateKeyUpdate()
                .set(DEPARTMENT_ENTITY_ACCESS.AUTHOR_ACCESS_ID, access.access.authorAccess.value.toByte())
                .set(DEPARTMENT_ENTITY_ACCESS.RESPONSIBLE_ACCESS_ID, access.access.responsibleAccess.value.toByte())
                .set(DEPARTMENT_ENTITY_ACCESS.FOLLOWER_ACCESS_ID, access.access.followerAccess.value.toByte())
                .execute()
        }

        return access
    }

    override fun getDepartmentAttributesAccesses(
        departmentId: UUID,
        entitiesAttributesIds: Collection<UUID>
    ): Collection<DepartmentAttributeAccess> {
        return jooqConfiguration.dsl()
            .select(*DEPARTMENT_ATTRIBUTE_ACCESS.fields())
            .where(DEPARTMENT_ATTRIBUTE_ACCESS.DEPARTMENT_ID.eq(departmentId))
            .and(DEPARTMENT_ATTRIBUTE_ACCESS.ENTITY_ATTRIBUTE_ID.`in`(entitiesAttributesIds))
            .fetchInto(DepartmentAttributeAccess::class.java)
    }

    override fun updateDepartmentAttributeAccess(access: DepartmentAttributeAccess): DepartmentAttributeAccess {
        jooqConfiguration.dsl().transaction { cfg ->
            cfg.dsl()
                .insertInto(
                    DEPARTMENT_ATTRIBUTE_ACCESS,
                    DEPARTMENT_ATTRIBUTE_ACCESS.DEPARTMENT_ID,
                    DEPARTMENT_ATTRIBUTE_ACCESS.ENTITY_ATTRIBUTE_ID,
                    DEPARTMENT_ATTRIBUTE_ACCESS.AUTHOR_ACCESS_ID,
                    DEPARTMENT_ATTRIBUTE_ACCESS.RESPONSIBLE_ACCESS_ID,
                    DEPARTMENT_ATTRIBUTE_ACCESS.FOLLOWER_ACCESS_ID,
                )
                .values(
                    access.departmentId,
                    access.entityAttributeId,
                    access.access.authorAccess.value.toByte(),
                    access.access.responsibleAccess.value.toByte(),
                    access.access.followerAccess.value.toByte()
                )
                .onDuplicateKeyUpdate()
                .set(DEPARTMENT_ATTRIBUTE_ACCESS.AUTHOR_ACCESS_ID, access.access.authorAccess.value.toByte())
                .set(DEPARTMENT_ATTRIBUTE_ACCESS.RESPONSIBLE_ACCESS_ID, access.access.responsibleAccess.value.toByte())
                .set(DEPARTMENT_ATTRIBUTE_ACCESS.FOLLOWER_ACCESS_ID, access.access.followerAccess.value.toByte())
                .execute()
        }

        return access
    }
}
