package ru.yandex.crm.apphost.kotlin.handlers.aggregator.aggregator.jooq

import org.jooq.Configuration
import org.jooq.Record5
import org.jooq.Select
import org.jooq.impl.DSL.and
import org.jooq.impl.DSL.max
import org.jooq.impl.DSL.min
import org.jooq.impl.DSL.row
import org.jooq.impl.DSL.select
import ru.yandex.crm.apphost.kotlin.handlers.aggregator.aggregator.AggregatorManager
import ru.yandex.crm.apphost.kotlin.handlers.aggregator.aggregator.model.FlatConnection
import ru.yandex.crm.apphost.kotlin.handlers.aggregator.aggregator.model.LinkFilter
import ru.yandex.crm.apphost.kotlin.handlers.aggregator.aggregator.model.LinkedEntities
import ru.yandex.crm.generated.database.aggregator.aggregator.tables.references.FLAT_CONNECTION
import java.util.UUID

class JooqAggregatorManager(
    private val jooqConfiguration: Configuration
) : AggregatorManager {

    override fun createLinks(links: Collection<FlatConnection>) {
        if (links.isEmpty()) {
            return
        }

        jooqConfiguration.dsl().transaction { cfg ->
            var insertCommand = cfg.dsl()
                .insertInto(
                    FLAT_CONNECTION,
                    FLAT_CONNECTION.ORGANIZATION_ID,
                    FLAT_CONNECTION.FIRST_META_ID,
                    FLAT_CONNECTION.FIRST_RECORD_INTERNAL_ID,
                    FLAT_CONNECTION.SECOND_META_ID,
                    FLAT_CONNECTION.SECOND_RECORD_INTERNAL_ID
                )
            for (link in links) {
                insertCommand = insertCommand
                    .values(
                        link.organizationId,
                        link.firstMetaId,
                        link.firstRecordInternalId,
                        link.secondMetaId,
                        link.secondRecordInternalId
                    )
            }
            insertCommand
                .onDuplicateKeyIgnore()
                .execute()
        }
    }

    override fun deleteLinks(links: Collection<FlatConnection>) {
        if (links.isEmpty()) {
            return
        }

        jooqConfiguration.dsl().transaction { cfg ->
            val linkRows = links.map {
                row(
                    it.organizationId,
                    it.firstMetaId,
                    it.firstRecordInternalId,
                    it.secondMetaId,
                    it.secondRecordInternalId
                )
            }.toTypedArray()

            cfg.dsl().delete(FLAT_CONNECTION)
                .where(
                    row(
                        FLAT_CONNECTION.ORGANIZATION_ID,
                        FLAT_CONNECTION.FIRST_META_ID,
                        FLAT_CONNECTION.FIRST_RECORD_INTERNAL_ID,
                        FLAT_CONNECTION.SECOND_META_ID,
                        FLAT_CONNECTION.SECOND_RECORD_INTERNAL_ID
                    )
                        .`in`(*linkRows)
                )
                .execute()
        }
    }

    override fun getLinks(filters: Collection<LinkFilter>, organizationId: Long, pageSize: Int): Collection<LinkedEntities> {
        var resultCommand: Select<Record5<UUID?, Long?, UUID?, Long?, Long?>>? = null

        val filtersByFirst = filters.filter { it.isFilterByFirst }
        if (filtersByFirst.isNotEmpty()) {
            resultCommand = select(
                FLAT_CONNECTION.FIRST_META_ID.`as`(LinkedEntities::rootMetaId.name),
                FLAT_CONNECTION.FIRST_RECORD_INTERNAL_ID.`as`(LinkedEntities::rootInternalRecordId.name),
                FLAT_CONNECTION.SECOND_META_ID.`as`(LinkedEntities::targetMetaId.name),
                min(FLAT_CONNECTION.SECOND_RECORD_INTERNAL_ID).`as`(LinkedEntities::internalIdFrom.name),
                max(FLAT_CONNECTION.SECOND_RECORD_INTERNAL_ID).`as`(LinkedEntities::internalIdTo.name),
            ).from(FLAT_CONNECTION)
                .where(FLAT_CONNECTION.ORGANIZATION_ID.eq(organizationId))
                .and(filtersByFirst.map {
                    val condition = and(FLAT_CONNECTION.FIRST_META_ID.eq(it.firstMetaId))
                        .and(FLAT_CONNECTION.FIRST_RECORD_INTERNAL_ID.eq(it.firstRecordInternalId))

                    if (it.secondMetaId != null) {
                        return@map condition.and(FLAT_CONNECTION.SECOND_META_ID.eq(it.secondMetaId))
                    }
                    return@map condition
                }.reduce { first, second -> first.or(second) })
                .groupBy(
                    FLAT_CONNECTION.FIRST_META_ID,
                    FLAT_CONNECTION.FIRST_RECORD_INTERNAL_ID,
                    FLAT_CONNECTION.SECOND_META_ID,
                    FLAT_CONNECTION.SECOND_RECORD_INTERNAL_ID / pageSize
                )
        }

        val filtersBySecond = filters.filter { !it.isFilterByFirst }
        if (filtersBySecond.isNotEmpty()) {
            val selectBySecond = select(
                FLAT_CONNECTION.SECOND_META_ID.`as`(LinkedEntities::rootMetaId.name),
                FLAT_CONNECTION.SECOND_RECORD_INTERNAL_ID.`as`(LinkedEntities::rootInternalRecordId.name),
                FLAT_CONNECTION.FIRST_META_ID.`as`(LinkedEntities::targetMetaId.name),
                min(FLAT_CONNECTION.FIRST_RECORD_INTERNAL_ID).`as`(LinkedEntities::internalIdFrom.name),
                max(FLAT_CONNECTION.FIRST_RECORD_INTERNAL_ID).`as`(LinkedEntities::internalIdTo.name),
            ).from(FLAT_CONNECTION)
                .where(FLAT_CONNECTION.ORGANIZATION_ID.eq(organizationId))
                .and(filtersBySecond.map {
                    val condition= and(FLAT_CONNECTION.SECOND_META_ID.eq(it.secondMetaId))
                        .and(FLAT_CONNECTION.SECOND_RECORD_INTERNAL_ID.eq(it.secondRecordInternalId))
                    if (it.firstMetaId != null) {
                        return@map condition.and(FLAT_CONNECTION.FIRST_META_ID.eq(it.firstMetaId))
                    }
                    return@map condition
                }.reduce { first, second -> first.or(second) })
                .groupBy(
                    FLAT_CONNECTION.SECOND_META_ID,
                    FLAT_CONNECTION.SECOND_RECORD_INTERNAL_ID,
                    FLAT_CONNECTION.FIRST_META_ID,
                    FLAT_CONNECTION.FIRST_RECORD_INTERNAL_ID / pageSize
                )
            resultCommand = resultCommand?.unionAll(selectBySecond) ?: selectBySecond
        }

        return jooqConfiguration.dsl().fetch(resultCommand)
            .into(LinkedEntities::class.java)
    }
}
