package ru.yandex.crm.apphost.kotlin.handlers.aggregator.aggregator.mappers.impl

import ru.yandex.crm.apphost.kotlin.handlers.aggregator.aggregator.mappers.LinkMapper
import ru.yandex.crm.apphost.kotlin.handlers.aggregator.aggregator.model.FlatConnection
import ru.yandex.crm.apphost.kotlin.handlers.aggregator.aggregator.model.LinkFilter
import ru.yandex.crm.apphost.kotlin.handlers.aggregator.aggregator.model.LinkedEntities
import ru.yandex.crm.apphost.proto.Aggregator
import ru.yandex.crm.apphost.proto.Aggregator.LinkFilter as ProtoLinkFilter
import ru.yandex.crm.apphost.proto.Aggregator.LinkedEntities as ProtoLinkedEntities
import java.util.UUID

class LinkMapperImpl: LinkMapper {
    override fun toFlatConnections(
        links: Collection<Aggregator.Link>,
        organizationId: Long
    ): Collection<FlatConnection> {
        return links.map {
            toFlatConnection(it, organizationId)
        }
    }

    override fun toLinkFilters(protoFilters: Collection<ProtoLinkFilter>): Collection<LinkFilter> {
        return protoFilters.map {
            toLinkFilter(it)
        }
    }

    override fun toProtoLinkedEntities(linkedEntities: Collection<LinkedEntities>): Collection<ProtoLinkedEntities> {
        return linkedEntities
            .groupBy { it.toKey() }
            .map {
                ProtoLinkedEntities.newBuilder()
                    .setRootMetaId(it.key.rootMetaId.toString())
                    .setRootInternalRecordId(it.key.rootInternalRecordId)
                    .setTargetMetaId(it.key.targetMetaId.toString())
                    .addAllIntervals(it.value.map { linkedEntities ->
                        Aggregator.LinkedEntities.FromTo.newBuilder()
                            .setInternalIdFrom(linkedEntities.internalIdFrom)
                            .setInternalIdTo(linkedEntities.internalIdTo)

                            .build()
                    })
                    .build()
            }
    }

    private fun toFlatConnection(link: Aggregator.Link, organizationId: Long): FlatConnection {
        return FlatConnection(
            organizationId,
            UUID.fromString(link.first.metaId),
            link.first.recordInternalId,
            UUID.fromString(link.second.metaId),
            link.second.recordInternalId
        )
    }

    private fun toLinkFilter(protoFilter: Aggregator.LinkFilter): LinkFilter {
        return if (protoFilter.hasFirst()) {
            LinkFilter(
                firstMetaId = UUID.fromString(protoFilter.first.entityId.metaId),
                firstRecordInternalId = protoFilter.first.entityId.recordInternalId,
                secondMetaId = if (protoFilter.first.hasLinkedMetaId()) UUID.fromString(protoFilter.first.linkedMetaId) else null,
                isFilterByFirst = true
            )
        } else {
            LinkFilter(
                secondMetaId = UUID.fromString(protoFilter.second.entityId.metaId),
                secondRecordInternalId = protoFilter.second.entityId.recordInternalId,
                firstMetaId = if (protoFilter.second.hasLinkedMetaId()) UUID.fromString(protoFilter.second.linkedMetaId) else null,
                isFilterByFirst = false
            )
        }
    }

    private data class LinkedEntitiesKey(
        val rootMetaId: UUID,
        val rootInternalRecordId: Long,
        val targetMetaId: UUID,
    )

    private fun LinkedEntities.toKey() = LinkedEntitiesKey(rootMetaId, rootInternalRecordId, targetMetaId)
}
