package ru.yandex.crm.apphost.kotlin.handlers.aggregator.ut.aggregator

import org.apache.commons.dbcp2.BasicDataSource
import org.jooq.Configuration
import org.jooq.SQLDialect
import org.jooq.conf.RenderQuotedNames
import org.jooq.conf.Settings
import org.jooq.impl.DefaultConfiguration
import org.junit.jupiter.api.AfterEach
import org.junit.jupiter.api.BeforeEach
import org.junit.jupiter.api.Test
import org.koin.core.context.startKoin
import org.koin.core.context.stopKoin
import org.koin.dsl.bind
import org.koin.dsl.module
import org.koin.test.KoinTest
import org.koin.test.inject
import ru.yandex.crm.apphost.kotlin.handlers.aggregator.aggregator.AggregatorManager
import ru.yandex.crm.apphost.kotlin.handlers.aggregator.aggregator.jooq.JooqAggregatorManager
import ru.yandex.crm.apphost.kotlin.handlers.aggregator.aggregator.model.LinkFilter
import ru.yandex.crm.generated.database.aggregator.aggregator.Aggregator
import ru.yandex.crm.generated.database.aggregator.aggregator.tables.references.FLAT_CONNECTION
import java.util.UUID
import javax.sql.DataSource
import kotlin.test.assertEquals

class AggregatorManagerTests : KoinTest {
    private val dataSource = BasicDataSource().apply {
        driverClassName = "org.h2.Driver"
        url = "jdbc:h2:mem:myDb;DB_CLOSE_DELAY=-1;INIT=CREATE SCHEMA IF NOT EXISTS ENTITYSTORAGE"
    }

    private fun createJooqConfiguration(dataSource: DataSource) = DefaultConfiguration().apply {
        setDataSource(dataSource)
        setSQLDialect(SQLDialect.H2)
        setSettings(Settings().withRenderQuotedNames(RenderQuotedNames.EXPLICIT_DEFAULT_UNQUOTED))
    }

    private val aggregatorManagerModule = module {
        single { dataSource } bind DataSource::class
        single { createJooqConfiguration(get()) } bind Configuration::class
        single { JooqAggregatorManager(get()) } bind AggregatorManager::class
    }

    @BeforeEach
    fun beforeEach() {
        startKoin {
            modules(
                aggregatorManagerModule
            )
        }
        generateTestData()
    }

    @AfterEach
    fun afterEach() {
        deleteTestData()
        stopKoin()
    }

    @Test
    fun `should return one internal when filter by first with linked meta`() {
        val aggregatorManager by inject<AggregatorManager>()
        val linkFilter = LinkFilter(
            firstMetaId = UUID(0, 1),
            firstRecordInternalId = 1,
            secondMetaId = UUID(0, 2),
            isFilterByFirst = true
        )

        val linkedEntities = aggregatorManager.getLinks(listOf(linkFilter), 1)

        assertEquals(1, linkedEntities.size)
    }

    @Test
    fun `should return 2 internals when filter by first`() {
        val aggregatorManager by inject<AggregatorManager>()
        val linkFilter = LinkFilter(
            firstMetaId = UUID(0, 1),
            firstRecordInternalId = 1,
            isFilterByFirst = true
        )

        val linkedEntities = aggregatorManager.getLinks(listOf(linkFilter), 1)

        assertEquals(2, linkedEntities.size)
    }

    @Test
    fun `should return one internal when filter by second with linked meta`() {
        val aggregatorManager by inject<AggregatorManager>()
        val linkFilter = LinkFilter(
            secondMetaId = UUID(0, 3),
            secondRecordInternalId = 1,
            firstMetaId = UUID(0, 1),
            isFilterByFirst = false
        )

        val linkedEntities = aggregatorManager.getLinks(listOf(linkFilter), 1)

        assertEquals(1, linkedEntities.size)
    }

    @Test
    fun `should return 2 internal when filter by second`() {
        val aggregatorManager by inject<AggregatorManager>()
        val linkFilter = LinkFilter(
            secondMetaId = UUID(0, 3),
            secondRecordInternalId = 1,
            isFilterByFirst = false
        )

        val linkedEntities = aggregatorManager.getLinks(listOf(linkFilter), 1)

        assertEquals(2, linkedEntities.size)
    }

    @Test
    fun `should return 2 internal for 2 filters for different linked meta`() {
        val aggregatorManager by inject<AggregatorManager>()
        val linkFilters = listOf(
            LinkFilter(
                secondMetaId = UUID(0, 3),
                secondRecordInternalId = 1,
                firstMetaId = UUID(0, 1),
                isFilterByFirst = false
            ),
            LinkFilter(
                secondMetaId = UUID(0, 3),
                secondRecordInternalId = 1,
                firstMetaId = UUID(0, 2),
                isFilterByFirst = false
            )
        )

        val linkedEntities = aggregatorManager.getLinks(linkFilters, 1)

        assertEquals(2, linkedEntities.size)
    }

    @Test
    fun `should return 2 internal for 2 filters for different records`() {
        val aggregatorManager by inject<AggregatorManager>()
        val linkFilters = listOf(
            LinkFilter(
                secondMetaId = UUID(0, 3),
                secondRecordInternalId = 1,
                firstMetaId = UUID(0, 1),
                isFilterByFirst = false
            ),
            LinkFilter(
                secondMetaId = UUID(0, 3),
                secondRecordInternalId = 2,
                firstMetaId = UUID(0, 1),
                isFilterByFirst = false
            )
        )

        val linkedEntities = aggregatorManager.getLinks(linkFilters, 1)

        assertEquals(2, linkedEntities.size)
    }

    @Test
    fun `should return 2 internal for 2 first and second filters`() {
        val aggregatorManager by inject<AggregatorManager>()
        val linkFilters = listOf(
            LinkFilter(
                secondMetaId = UUID(0, 3),
                secondRecordInternalId = 1,
                firstMetaId = UUID(0, 1),
                isFilterByFirst = false
            ),
            LinkFilter(
                firstMetaId = UUID(0, 1),
                firstRecordInternalId = 1,
                secondMetaId = UUID(0, 2),
                isFilterByFirst = true
            )
        )

        val linkedEntities = aggregatorManager.getLinks(linkFilters, 1)

        assertEquals(2, linkedEntities.size)
    }

    private fun generateTestData() {
        val jooqConfiguration by inject<Configuration>()
        jooqConfiguration.dsl().createSchema(Aggregator.AGGREGATOR)
            .execute()
        jooqConfiguration.dsl().createTable(FLAT_CONNECTION)
            .column(FLAT_CONNECTION.ID)
            .column(FLAT_CONNECTION.ORGANIZATION_ID)
            .column(FLAT_CONNECTION.FIRST_META_ID)
            .column(FLAT_CONNECTION.FIRST_RECORD_INTERNAL_ID)
            .column(FLAT_CONNECTION.SECOND_META_ID)
            .column(FLAT_CONNECTION.SECOND_RECORD_INTERNAL_ID)
            .execute()

        val firstMetaId = UUID(0, 1)
        val secondMetaId = UUID(0, 2)
        val thirdMetaId = UUID(0, 3)

        var command = jooqConfiguration.dsl().insertInto(
            FLAT_CONNECTION,
            FLAT_CONNECTION.ORGANIZATION_ID,
            FLAT_CONNECTION.FIRST_META_ID,
            FLAT_CONNECTION.FIRST_RECORD_INTERNAL_ID,
            FLAT_CONNECTION.SECOND_META_ID,
            FLAT_CONNECTION.SECOND_RECORD_INTERNAL_ID
        )

        for (i in 1..100L) {
            command = command
                .values(1L, firstMetaId, i, secondMetaId, i)
                .values(1L, firstMetaId, i, thirdMetaId, i)
                .values(1L, secondMetaId, i, thirdMetaId, i)
        }
        command.execute()
    }

    private fun deleteTestData() {
        val jooqConfiguration by inject<Configuration>()
        jooqConfiguration.dsl().dropSchema(Aggregator.AGGREGATOR)
            .cascade()
            .execute()
    }
}
