package ru.yandex.crm.apphost.kotlin.handlers.auditor

import com.google.common.net.HostAndPort
import com.google.common.util.concurrent.ThreadFactoryBuilder
import org.koin.dsl.bind
import org.koin.dsl.module
import ru.yandex.crm.apphost.kotlin.handlers.auditor.service.LoggerService
import ru.yandex.crm.apphost.kotlin.handlers.auditor.service.config.ClientConfiguration
import ru.yandex.crm.apphost.kotlin.handlers.auditor.service.config.SessionConfiguration
import ru.yandex.crm.apphost.kotlin.handlers.auditor.service.impl.UnifiedAgentLoggerService
import ru.yandex.crm.apphost.kotlin.handlers.auditor.service.mappers.LogProtoModelsMapper
import ru.yandex.crm.apphost.kotlin.handlers.auditor.service.mappers.impl.LogProtoModelsMapperImpl
import ru.yandex.crm.library.kotlin.service.HostBuilder
import ru.yandex.crm.library.kotlin.service.addOptions
import ru.yandex.logbroker.agent.client.Client
import ru.yandex.logbroker.agent.client.Session
import ru.yandex.web.apphost.api.grpc.AppHostTypedServant
import java.time.Duration
import java.util.concurrent.Executors

fun HostBuilder.withAuditorModule() = withModule(module {
    addOptions<ClientConfiguration>("unfiedAgent")
    addOptions<SessionConfiguration>("unfiedAgent")

    single {
        val clientConfig = get<ClientConfiguration>()

        Client.newClient(HostAndPort.fromParts(clientConfig.url, clientConfig.port))
            .setSharedSecretKey(clientConfig.sharedKey)
            .setCloseTimeout(Duration.ofSeconds(clientConfig.closeTimeoutSeconds))
            .setGrcpReconnectDelay(Duration.ofSeconds(clientConfig.reconnectDelaySeconds))
            .build()
    } bind Client::class

    single {
        val sessionConfig = get<SessionConfiguration>()
        val client = get<Client>()
        val threadPool = Executors.newScheduledThreadPool(
            sessionConfig.threadPoolSize,
            ThreadFactoryBuilder()
                .setDaemon(true)
                .setNameFormat("UnifiedAgentExecutor-%d")
                .build()
        )

        client.newSession()
            .setThreadPool(threadPool)
            .build()
    } bind Session::class

    single { LogProtoModelsMapperImpl() } bind LogProtoModelsMapper::class

    single { UnifiedAgentLoggerService(get()) } bind LoggerService::class

    single { GrpcAuditorService(get(), get()) } bind AppHostTypedServant::class
})
