package ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.repository.impl

import org.hibernate.Session
import ru.yandex.crm.apphost.kotlin.dal.departmentmanager.Department
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.repository.DepartmentRepository
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.repository.DepartmentRepository.DepartmentUsersCount
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateRepository
import java.math.BigInteger
import java.util.*

class DepartmentRepositoryImpl(
    session: Session
) : HibernateRepository<Department, UUID>(session, Department::class.java), DepartmentRepository {

    override fun findAll(includeDeleted: Boolean): List<Department> {
        val queryString = StringBuilder("FROM Department")
        if (!includeDeleted) {
            queryString.append(" WHERE is_active = true")
        }
        val query = session.createQuery(queryString.toString(), Department::class.java)
        return query.resultList
    }

    override fun findDepartmentById(departmentId: UUID, organizationId: Long): Department? {
        val queryString = "FROM Department WHERE organization_id = :organizationId " +
            "AND id = :departmentId"
        val query = session.createQuery(queryString, Department::class.java).apply {
            setParameter("organizationId", organizationId)
            setParameter("departmentId", departmentId)
        }
        return query.resultList.firstOrNull()
    }

    override fun getAllOrganizationDepartments(organizationId: Long, includeDeleted: Boolean): List<Department> {
        val queryString = StringBuilder("FROM Department WHERE organization_id = :organizationId")
        if (!includeDeleted) {
            queryString.append(" AND is_active = true")
        }
        val query = session.createQuery(queryString.toString(), Department::class.java)
        query.setParameter("organizationId", organizationId)
        return query.resultList
    }

    override fun listOrganizationDepartmentsByPattern(
        organizationId: Long,
        includeDeleted: Boolean,
        pattern: String,
        languageCode: String
    ): List<Department> {
        val queryString = StringBuilder("SELECT DISTINCT(d) FROM DepartmentName n LEFT JOIN Department d ON d = n.id.department WHERE d.organizationId = :organizationId " +
            "AND lower(n.name) like :pattern AND n.id.languageCode = :locale")
        if (!includeDeleted) {
            queryString.append(" AND d.isActive = true")
        }
        val query = session.createQuery(queryString.toString(), Department::class.java)
        query.setParameter("organizationId", organizationId)
        query.setParameter("pattern", "%${pattern.lowercase()}%")
        query.setParameter("locale", languageCode)
        return query.resultList
    }

    override fun countDepartmentUsers(organizationId: Long, departmentId: UUID): DepartmentUsersCount {
        val queryString = "SELECT COUNT(*) FROM departmentmanager.department_user d " +
            "WHERE d.organization_id = :organizationId " +
            "AND d.department_id = :departmentId"
        val query = session.createNativeQuery(queryString).apply {
            setParameter("departmentId", departmentId)
            setParameter("organizationId", organizationId)
        }
        val count = (query.singleResult as BigInteger).toInt()
        return DepartmentUsersCount(departmentId.toString(), count)
    }

    override fun countAllOrganizationDepartmentUsers(organizationId: Long): List<DepartmentUsersCount> {
        val queryString = "SELECT COUNT(*), CAST(d.department_id AS varchar) dep_id FROM departmentmanager.department_user d " +
            "WHERE d.organization_id = :organizationId " +
            "GROUP BY d.department_id"
        val query = session.createNativeQuery(queryString).apply {
            setParameter("organizationId", organizationId)
        }
        val result = query.resultList
        return result.map { rawResultSet ->
            val resultSet = rawResultSet as Array<*>
            val count = (resultSet[0] as BigInteger).toInt()
            val departmentId = resultSet[1] as String

            DepartmentUsersCount(departmentId, count)
        }
    }

    override fun departmentContainsUser(departmentId: UUID, organizationId: Long, userId: Long): Boolean {
        val queryString = "SELECT EXISTS( " +
            "SELECT user_id FROM departmentmanager.department_user d " +
            "WHERE d.user_id = :userId AND d.organization_id = :organizationId AND d.department_id = :departmentId" +
            " )"
        val query = session.createNativeQuery(queryString).apply {
            setParameter("userId", userId)
            setParameter("organizationId", organizationId)
            setParameter("departmentId", departmentId)
        }
        return query.singleResult as Boolean
    }
}
