package ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.impl

import ru.yandex.crm.apphost.kotlin.dal.departmentmanager.Department
import ru.yandex.crm.apphost.kotlin.dal.departmentmanager.DepartmentResponsibleUser
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.repository.DepartmentRepository
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.repository.DepartmentResponsibleUserRepository
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.repository.OrganizationRepository
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.DepartmentService
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.components.DepartmentNameComponent
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.mappers.DepartmentEntityMapper
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.mappers.ProtoDepartment
import ru.yandex.crm.library.kotlin.database.hibernate.getRepository
import ru.yandex.crm.library.kotlin.database.hibernate.transaction
import ru.yandex.crm.proto.gallifrey.departmentmanager.Departmentmanager
import ru.yandex.crm.proto.gallifrey.departmentmanager.Departmentmanager.DepartmentData
import ru.yandex.crm.proto.gallifrey.departmentmanager.Departmentmanager.UpdateDepartmentData
import java.util.*

class DepartmentServiceImpl(
    private val entityMapper: DepartmentEntityMapper,
    private val nameComponent: DepartmentNameComponent,
) : DepartmentService {

    override fun getAllDepartments(includeDeleted: Boolean): List<ProtoDepartment> {
        val departments = transaction {
            val repository = getRepository<DepartmentRepository>()
            repository.findAll(includeDeleted)
        }
        return departments.map { entityMapper.toProtobufModel(it) }
    }

    override fun getDepartmentById(departmentId: String, organizationId: Long): ProtoDepartment {
        val department = transaction {
            val repository = getRepository<DepartmentRepository>()
            val departmentUUID = UUID.fromString(departmentId)
            repository.findDepartmentById(departmentUUID, organizationId)
                ?: error("Department with id: $departmentId not found")
        }
        return entityMapper.toProtobufModel(department)
    }

    override fun getOrganizationDepartments(organizationId: Long, includeDeleted: Boolean): List<ProtoDepartment> {
        val departments = transaction {
            val repository = getRepository<DepartmentRepository>()
            repository.getAllOrganizationDepartments(organizationId, includeDeleted)
        }
        return departments.map { entityMapper.toProtobufModel(it) }
    }

    override fun getOrganizationDepartmentsByPattern(
        organizationId: Long,
        includeDeleted: Boolean,
        pattern: String,
        languageCode: String
    ): List<Departmentmanager.Department> {
        val departments = transaction {
            val repository = getRepository<DepartmentRepository>()
            repository.listOrganizationDepartmentsByPattern(organizationId, includeDeleted, pattern, languageCode)
        }
        return departments.map { entityMapper.toProtobufModel(it) }
    }

    override fun createDepartment(departmentData: DepartmentData, responsileUserIds: List<Long>): ProtoDepartment {
        val department = transaction {
            val departmentRepository = getRepository<DepartmentRepository>()
            val organizationRepository = getRepository<OrganizationRepository>()

            val parentDepartmentId = departmentData.parentDepartmentId
            val organizationId = departmentData.organizationId

            val parentDepartment = departmentRepository.findOne(UUID.fromString(parentDepartmentId))
            val organization = organizationRepository.findOne(organizationId)
                ?: error("Organization with id: $organizationId not exists")
            var department = Department(
                organizationId = organizationId,
                parentDepartment = parentDepartment
            )

            department = departmentRepository.save(department)

            val departmentNames = departmentData.namesList.map { name ->
                nameComponent.createDepartmentName(department, organizationId, name)
            }.toMutableSet()
            department.names = departmentNames

            val departmentResponsibleRepository = getRepository<DepartmentResponsibleUserRepository>()

            if (responsileUserIds.isEmpty())
                error("At least one responsible must be provided")

            responsileUserIds.forEach { userId ->
                val departmentUser = DepartmentResponsibleUser(
                    user = userId,
                    department = department,
                    organization = organizationId,
                )
                department.responsibleUsers.add(departmentUser)
                departmentResponsibleRepository.save(departmentUser)
            }
            departmentRepository.save(department)
        }

        return entityMapper.toProtobufModel(department)
    }

    override fun updateDepartment(
        updateData: UpdateDepartmentData,
        organizationId: Long,
        departmentId: String
    ): ProtoDepartment {
        val department = transaction {
            val departmentRepository = getRepository<DepartmentRepository>()
            val organizationRepository = getRepository<OrganizationRepository>()

            val departmentUUID = UUID.fromString(departmentId)
            val targetDepartment = departmentRepository.findDepartmentById(departmentUUID, organizationId)
                ?: error("Department with id: $departmentUUID not found")

            if (updateData.hasNewOrganizationId()) {
                val newOrganizationId = updateData.newOrganizationId
                val newOrganization = organizationRepository.findOne(newOrganizationId)
                    ?: error("Organization with id: $newOrganizationId not exists")
                targetDepartment.organizationId = newOrganizationId
            }

            if (updateData.hasNewParentDepartmentId()) {
                val newParentDepartmentId = updateData.newParentDepartmentId
                val newParentDepartmentUUID = UUID.fromString(newParentDepartmentId)
                val newParentDepartment = departmentRepository.findDepartmentById(newParentDepartmentUUID, organizationId)
                    ?: error("Department with id: $newParentDepartmentId not found")
                targetDepartment.parentDepartment = newParentDepartment
            }

            val currentNames = targetDepartment.names
            val newNames = updateData.newNamesList.map { newName ->
                nameComponent.createDepartmentName(targetDepartment, organizationId, newName)
            }.toMutableSet()
            val updatedNames = nameComponent.updateDepartmentNames(currentNames, newNames)
            targetDepartment.names.clear()
            targetDepartment.names.addAll(updatedNames)

            departmentRepository.save(targetDepartment)
        }

        return entityMapper.toProtobufModel(department)
    }

    override fun archiveDepartment(organizationId: Long, departmentId: String) {
        transaction {
            val repository = getRepository<DepartmentRepository>()
            val departmentUUID = UUID.fromString(departmentId)
            val department = repository.findDepartmentById(departmentUUID, organizationId)
                ?: error("Department with id: $departmentId not found")
            department.isActive = false
            repository.save(department)
        }
    }
}
