package ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.impl

import org.hibernate.Hibernate
import ru.yandex.crm.apphost.kotlin.dal.departmentmanager.Department
import ru.yandex.crm.apphost.kotlin.dal.departmentmanager.DepartmentResponsibleUser
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.repository.DepartmentRepository
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.repository.DepartmentResponsibleUserRepository
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.repository.DepartmentUserRepository
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.DepartmentUsersService
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.DepartmentUsersService.User
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.mappers.DepartmentUsersCountMapper
import ru.yandex.crm.library.kotlin.database.hibernate.getRepository
import ru.yandex.crm.library.kotlin.database.hibernate.transaction
import ru.yandex.crm.proto.gallifrey.departmentmanager.Departmentmanager.DepartmentUsersCount
import java.util.*

class DepartmentUsersServiceImpl(
    private val mapper: DepartmentUsersCountMapper,
): DepartmentUsersService {

    override fun getDepartmentUsersList(
        organizationId: Long, departmentId: String
    ): List<User> {
        val members = transaction {
            val repository = getRepository<DepartmentUserRepository>()
            val departmentUUID = UUID.fromString(departmentId)
            val users = repository.findAll(
                "FROM DepartmentUser WHERE organization_id = :organizationId AND department_id = :departmentId",
                "organizationId" to organizationId,
                "departmentId" to departmentUUID
            )

            users.map { it.user }
        }
        return members.map { User(it.id!!) }
    }

    override fun getDepartmentUsersCount(organizationId: Long, departmentId: String): DepartmentUsersCount {
        val membersCount =  transaction {
            val repository = getRepository<DepartmentRepository>()
            val departmentUUID = UUID.fromString(departmentId)
            repository.countDepartmentUsers(organizationId, departmentUUID)
        }
        return mapper.toProtobufModel(membersCount, organizationId)
    }

    override fun getAllOrganizationDepartmentUsersCount(organizationId: Long): List<DepartmentUsersCount> {
        val membersCount =  transaction {
            val repository = getRepository<DepartmentRepository>()
            repository.countAllOrganizationDepartmentUsers(organizationId)
        }
        return membersCount.map { mapper.toProtobufModel(it, organizationId) }
    }

    override fun getDepartmentResponsibleList(
        organizationId: Long, departmentId: String, includeParentDepartments: Boolean
    ): List<User> {
        val responsible = transaction {
            val repository = getRepository<DepartmentRepository>()
            val responsibleRepository = getRepository<DepartmentResponsibleUserRepository>()

            val departmentUUID = UUID.fromString(departmentId)
            val organizationDepartments = repository.getAllOrganizationDepartments(organizationId, includeDeleted = false)
            var currentDepartment: Department? = organizationDepartments.find {it.id == departmentUUID }
                ?: error("Department with id: $departmentId not found")

            val branchDepartmentIds = mutableListOf<UUID>()

            if (includeParentDepartments) {
                while (currentDepartment != null) {
                    branchDepartmentIds.add(currentDepartment.id!!)
                    val parentId = currentDepartment.parentDepartment?.id
                    currentDepartment = organizationDepartments.find { it.id == parentId }
                }
            } else {
                branchDepartmentIds.add(currentDepartment?.id!!)
            }

            responsibleRepository.findAll(
                "FROM DepartmentResponsibleUser WHERE department_id in (:ids)", "ids" to branchDepartmentIds
            )
        }
        return responsible.map { User(it.user) }
    }

    private fun getDepartmentResponsibleUsers(department: Department): List<Long> {
        val departmentUsers = department.responsibleUsers
        Hibernate.initialize(departmentUsers)
        return departmentUsers.map { it.user }
    }
}
