package ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.ut.service

import org.apache.commons.dbcp2.BasicDataSource
import org.junit.jupiter.api.AfterEach
import org.junit.jupiter.api.BeforeEach
import org.junit.jupiter.api.Test
import org.junit.jupiter.api.assertThrows
import org.koin.core.context.startKoin
import org.koin.core.context.stopKoin
import org.koin.dsl.bind
import org.koin.dsl.module
import org.koin.test.KoinTest
import org.koin.test.inject
import ru.yandex.crm.apphost.kotlin.dal.departmentmanager.schema.DepartmentManagerSchema
import ru.yandex.crm.apphost.kotlin.dal.organizationmanager.schema.OrganizationManagerSchema
import ru.yandex.crm.apphost.kotlin.dal.usermanager.schema.UserManagerSchema
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.repository.DepartmentRepository
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.repository.DepartmentResponsibleUserRepository
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.repository.DepartmentUserRepository
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.repository.OrganizationRepository
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.DepartmentService
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.components.DepartmentNameComponent
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.components.impl.DepartmentNameComponentImpl
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.impl.DepartmentServiceImpl
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.mappers.DepartmentEntityMapper
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.mappers.impl.DepartmentEntityMapperImpl
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.ut.repository.MockDepartmentRepository
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.ut.repository.MockDepartmentResponsibleUserRepository
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.ut.repository.MockDepartmentUserRepository
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.ut.repository.MockOrganizationRepository
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateConfig
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateSchema
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateSessionManager
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateSessionManagerImpl
import ru.yandex.crm.proto.gallifrey.departmentmanager.Departmentmanager.*
import javax.sql.DataSource
import kotlin.test.assertEquals
import kotlin.test.assertNotNull
import kotlin.test.assertTrue

private val hibernateConfig = HibernateConfig().apply {
    dialect = "org.hibernate.dialect.H2Dialect"
    showSql = true
    hbm2ddlAuto = "none"
}

private val dataSource = BasicDataSource().apply {
    driverClassName = "org.h2.Driver"
    url = "jdbc:h2:mem:myDb;DB_CLOSE_DELAY=-1"
}

private val serviceModule = module {
    single { hibernateConfig } bind HibernateConfig::class
    single { dataSource } bind DataSource::class
    single { DepartmentManagerSchema() } bind HibernateSchema::class
    single { UserManagerSchema() } bind HibernateSchema::class
    single { OrganizationManagerSchema() } bind HibernateSchema::class
    single { HibernateSessionManagerImpl(get(), get(), getAll()) } bind HibernateSessionManager::class

    factory { MockDepartmentRepository() } bind DepartmentRepository::class
    factory { MockOrganizationRepository() } bind OrganizationRepository::class
    single { MockDepartmentResponsibleUserRepository() } bind DepartmentResponsibleUserRepository::class
    single { MockDepartmentUserRepository() } bind DepartmentUserRepository::class

    single { DepartmentServiceImpl(get(), get()) } bind DepartmentService::class
    single { DepartmentEntityMapperImpl() } bind DepartmentEntityMapper::class
    single { DepartmentNameComponentImpl() } bind DepartmentNameComponent::class
}

class DepartmentServiceTests : KoinTest {

    @BeforeEach
    fun beforeEach() {
        startKoin {
            modules(serviceModule)
        }
    }

    @AfterEach
    fun afterEach() {
        stopKoin()
    }

    @Test
    fun `get all departments`() {
        val service by inject<DepartmentService>()

        val departments = service.getAllDepartments(includeDeleted = true)

        assertTrue(departments.size == 4)
    }

    @Test
    fun `get only active departments`() {
        val service by inject<DepartmentService>()

        val departments = service.getAllDepartments(includeDeleted = false)

        assertTrue(departments.size == 3)
    }

    @Test
    fun `get department by id`() {
        val service by inject<DepartmentService>()

        val department1 = service.getDepartmentById("327ba20a-1d9c-4d73-8029-889656957e36", 1L)
        val department2 = service.getDepartmentById("49d51155-c672-4b28-896a-8a9bd39ab282", 1L)
        val department3 = service.getDepartmentById("1655d06f-0cb6-41d0-82e7-88cfc189b2c5", 1L)

        assertEquals("327ba20a-1d9c-4d73-8029-889656957e36", department1.id)
        assertEquals("49d51155-c672-4b28-896a-8a9bd39ab282", department2.id)
        assertEquals("1655d06f-0cb6-41d0-82e7-88cfc189b2c5", department3.id)

        assertEquals(1L, department1.data.organizationId)
        assertEquals(1L, department2.data.organizationId)
        assertEquals(1L, department3.data.organizationId)

        assertTrue(department1.data.namesList.size == 1)
        assertTrue(department2.data.namesList.size == 1)
        assertTrue(department3.data.namesList.size == 1)
    }

    @Test
    fun `get organization departments`() {
        val service by inject<DepartmentService>()

        val departments = service.getOrganizationDepartments(1L, includeDeleted = true)

        assertTrue(departments.size == 4)
    }

    @Test
    fun `get only active organization departments`() {
        val service by inject<DepartmentService>()

        val departments = service.getOrganizationDepartments(1L, includeDeleted = false)

        assertTrue(departments.size == 3)
    }

    @Test
    fun `create department`() {
        val service by inject<DepartmentService>()

        val name1 = DepartmentName.newBuilder()
            .setName("Имя")
            .setLanguageCode("ru")
            .build()
        val name2 = DepartmentName.newBuilder()
            .setName("Name")
            .setLanguageCode("en")
            .build()
        val createModel = DepartmentData.newBuilder()
            .setParentDepartmentId("1655d06f-0cb6-41d0-82e7-88cfc189b2c5")
            .setOrganizationId(1L)
            .addAllNames(listOf(name1, name2))
            .build()

        val department = service.createDepartment(createModel, listOf(1L))

        assertTrue(department.id.isNotBlank())

        assertEquals(1L, department.data.organizationId)
        assertEquals("1655d06f-0cb6-41d0-82e7-88cfc189b2c5", department.data.parentDepartmentId)
        assertTrue(department.data.namesList.size == 2)

        val departmentResponsibleUserRepository by inject<DepartmentResponsibleUserRepository>()
        val responsibleUser = departmentResponsibleUserRepository.findAll().firstOrNull { it.department.id.toString() == department.id}
        assertNotNull(responsibleUser)
        assertEquals(1L, responsibleUser.user)
    }

    @Test
    fun `create department with empty responsible causes error`() {
        val service by inject<DepartmentService>()

        val name1 = DepartmentName.newBuilder()
            .setName("Имя")
            .setLanguageCode("ru")
            .build()
        val name2 = DepartmentName.newBuilder()
            .setName("Name")
            .setLanguageCode("en")
            .build()
        val createModel = DepartmentData.newBuilder()
            .setParentDepartmentId("1655d06f-0cb6-41d0-82e7-88cfc189b2c5")
            .setOrganizationId(1L)
            .addAllNames(listOf(name1, name2))
            .build()

        assertThrows<IllegalStateException> { service.createDepartment(createModel, listOf()) }
    }

    @Test
    fun `update department`() {
        val service by inject<DepartmentService>()

        val organizationId = 1L
        val departmentId = "1655d06f-0cb6-41d0-82e7-88cfc189b2c5"
        val updateModel = UpdateDepartmentData.newBuilder()
            .setNewParentDepartmentId("49d51155-c672-4b28-896a-8a9bd39ab282")
            .build()

        val newDepartment = service.updateDepartment(updateModel, organizationId, departmentId)

        assertEquals(departmentId, newDepartment.id)
        assertEquals("49d51155-c672-4b28-896a-8a9bd39ab282", newDepartment.data.parentDepartmentId)
        assertEquals(organizationId, newDepartment.data.organizationId)

        assertTrue(newDepartment.data.namesList.size == 0)
    }

    @Test
    fun `archive department`() {
        val service by inject<DepartmentService>()

        val organizationId = 1L
        val departmentId = "1655d06f-0cb6-41d0-82e7-88cfc189b2c5"

        service.archiveDepartment(organizationId, departmentId)
    }
}
