package ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.ut.service

import org.apache.commons.dbcp2.BasicDataSource
import org.junit.jupiter.api.AfterEach
import org.junit.jupiter.api.BeforeEach
import org.junit.jupiter.api.Test
import org.koin.core.context.startKoin
import org.koin.core.context.stopKoin
import org.koin.dsl.bind
import org.koin.dsl.module
import org.koin.test.KoinTest
import org.koin.test.inject
import ru.yandex.crm.apphost.kotlin.dal.departmentmanager.schema.DepartmentManagerSchema
import ru.yandex.crm.apphost.kotlin.dal.organizationmanager.schema.OrganizationManagerSchema
import ru.yandex.crm.apphost.kotlin.dal.usermanager.schema.UserManagerSchema
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.repository.DepartmentRepository
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.repository.DepartmentUserRepository
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.repository.OrganizationRepository
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.DepartmentService
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.DepartmentUsersService
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.components.DepartmentNameComponent
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.components.impl.DepartmentNameComponentImpl
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.impl.DepartmentServiceImpl
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.impl.DepartmentUsersServiceImpl
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.mappers.DepartmentEntityMapper
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.mappers.DepartmentUsersCountMapper
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.mappers.impl.DepartmentEntityMapperImpl
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.mappers.impl.DepartmentUsersCountMapperImpl
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.ut.repository.MockDepartmentRepository
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.ut.repository.MockDepartmentUserRepository
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.ut.repository.MockOrganizationRepository
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateConfig
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateSchema
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateSessionManager
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateSessionManagerImpl
import javax.sql.DataSource
import kotlin.test.assertEquals
import kotlin.test.assertTrue

private val hibernateConfig = HibernateConfig().apply {
    dialect = "org.hibernate.dialect.H2Dialect"
    showSql = true
    hbm2ddlAuto = "none"
}

private val dataSource = BasicDataSource().apply {
    driverClassName = "org.h2.Driver"
    url = "jdbc:h2:mem:myDb;DB_CLOSE_DELAY=-1"
}

private val serviceModule = module {
    single { hibernateConfig } bind HibernateConfig::class
    single { dataSource } bind DataSource::class
    single { DepartmentManagerSchema() } bind HibernateSchema::class
    single { UserManagerSchema() } bind HibernateSchema::class
    single { OrganizationManagerSchema() } bind HibernateSchema::class
    single { HibernateSessionManagerImpl(get(), get(), getAll()) } bind HibernateSessionManager::class

    factory { MockDepartmentRepository() } bind DepartmentRepository::class
    factory { MockOrganizationRepository() } bind OrganizationRepository::class
    factory { MockDepartmentUserRepository() } bind DepartmentUserRepository::class

    single { DepartmentServiceImpl(get(), get()) } bind DepartmentService::class
    single { DepartmentUsersServiceImpl(get()) } bind DepartmentUsersService::class
    single { DepartmentEntityMapperImpl() } bind DepartmentEntityMapper::class
    single { DepartmentUsersCountMapperImpl() } bind DepartmentUsersCountMapper::class
    single { DepartmentNameComponentImpl() } bind DepartmentNameComponent::class
}

class DepartmentUsersServiceTest: KoinTest {

    @BeforeEach
    fun beforeEach() {
        startKoin {
            modules(serviceModule)
        }
    }

    @AfterEach
    fun afterEach() {
        stopKoin()
    }

    @Test
    fun `get departments members list`() {
        val service by inject<DepartmentUsersService>()

        val members1 = service.getDepartmentUsersList(1L, "327ba20a-1d9c-4d73-8029-889656957e36")
        val members2 = service.getDepartmentUsersList(1L, "49d51155-c672-4b28-896a-8a9bd39ab282")
        val members3 = service.getDepartmentUsersList(1L, "1655d06f-0cb6-41d0-82e7-88cfc189b2c5")

        assertTrue(members1.size == 1)
        assertTrue(members2.size == 1)
        assertTrue(members3.size == 1)
    }

    @Test
    fun `get departments members count`() {
        val service by inject<DepartmentUsersService>()

        val membersCount1 = service.getDepartmentUsersCount(1L, "327ba20a-1d9c-4d73-8029-889656957e36")
        val membersCount2 = service.getDepartmentUsersCount(1L, "49d51155-c672-4b28-896a-8a9bd39ab282")
        val membersCount3 = service.getDepartmentUsersCount(1L, "1655d06f-0cb6-41d0-82e7-88cfc189b2c5")

        assertTrue(membersCount1.usersCount == 1)
        assertTrue(membersCount1.departmentId == "327ba20a-1d9c-4d73-8029-889656957e36")
        assertTrue(membersCount1.organizationId == 1L)

        assertTrue(membersCount2.usersCount == 1)
        assertTrue(membersCount2.departmentId == "49d51155-c672-4b28-896a-8a9bd39ab282")
        assertTrue(membersCount2.organizationId == 1L)

        assertTrue(membersCount3.usersCount == 1)
        assertTrue(membersCount3.departmentId == "1655d06f-0cb6-41d0-82e7-88cfc189b2c5")
        assertTrue(membersCount3.organizationId == 1L)
    }

    @Test
    fun `get all organization departments members count`() {
        val service by inject<DepartmentUsersService>()

        val membersCount = service.getAllOrganizationDepartmentUsersCount(1L)

        assertEquals(4, membersCount.size)
        membersCount.forEach { count ->
            assertTrue(count.usersCount == 1)
            assertTrue(count.organizationId == 1L)
        }
    }
}
