package ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.ut.service

import org.apache.commons.dbcp2.BasicDataSource
import org.junit.jupiter.api.AfterEach
import org.junit.jupiter.api.BeforeEach
import org.junit.jupiter.api.Test
import org.koin.core.context.startKoin
import org.koin.core.context.stopKoin
import org.koin.dsl.bind
import org.koin.dsl.module
import org.koin.test.KoinTest
import org.koin.test.get
import ru.yandex.crm.apphost.kotlin.common.validator.ModelValidator
import ru.yandex.crm.apphost.kotlin.dal.departmentmanager.schema.DepartmentManagerSchema
import ru.yandex.crm.apphost.kotlin.dal.organizationmanager.Organization
import ru.yandex.crm.apphost.kotlin.dal.organizationmanager.schema.OrganizationManagerSchema
import ru.yandex.crm.apphost.kotlin.dal.usermanager.schema.UserManagerSchema
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.repository.DepartmentRepository
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.repository.DepartmentResponsibleUserRepository
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.repository.DepartmentUserRepository
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.repository.OrganizationRepository
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.repository.UserRepository
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.repository.impl.DepartmentRepositoryImpl
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.repository.impl.DepartmentResponsibleUserRepositoryImpl
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.repository.impl.DepartmentUserRepositoryImpl
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.repository.impl.OrganizationRepositoryImpl
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.DepartmentLinkerService
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.DepartmentService
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.DepartmentUsersService
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.components.DepartmentNameComponent
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.components.impl.DepartmentNameComponentImpl
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.impl.DepartmentLinkerServiceImpl
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.impl.DepartmentServiceImpl
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.impl.DepartmentUsersServiceImpl
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.mappers.DepartmentEntityMapper
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.mappers.DepartmentUsersCountMapper
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.mappers.impl.DepartmentEntityMapperImpl
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.service.mappers.impl.DepartmentUsersCountMapperImpl
import ru.yandex.crm.apphost.kotlin.handlers.departmentmanager.validators.UpdateDepartmentDataValidator
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateConfig
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateSchema
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateSessionManager
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateSessionManagerImpl
import ru.yandex.crm.library.kotlin.database.hibernate.getRepository
import ru.yandex.crm.library.kotlin.database.hibernate.transaction
import ru.yandex.crm.proto.gallifrey.departmentmanager.Departmentmanager
import ru.yandex.crm.proto.gallifrey.departmentmanager.Departmentmanager.DepartmentData
import javax.sql.DataSource
import kotlin.test.assertEquals

class FilterDepartmentsTests : KoinTest {
    protected open val hibernateConfig = HibernateConfig().apply {
        dialect = "org.hibernate.dialect.H2Dialect"
        showSql = true
        hbm2ddlAuto = "update"
    }

    protected open val dataSource = BasicDataSource().apply {
        driverClassName = "org.h2.Driver"
        url =
            "jdbc:h2:mem:myDb;DB_CLOSE_DELAY=-1;INIT=CREATE SCHEMA IF NOT EXISTS ORGANIZATIONMANAGER\\;" +
                "CREATE SCHEMA IF NOT EXISTS USERMANAGER\\;CREATE SCHEMA IF NOT EXISTS DEPARTMENTMANAGER"
    }

    private val departmentModule = module {
        single { hibernateConfig } bind HibernateConfig::class
        single { dataSource } bind DataSource::class
        single { DepartmentManagerSchema() } bind HibernateSchema::class
        single { OrganizationManagerSchema() } bind HibernateSchema::class
        single { UserManagerSchema() } bind HibernateSchema::class
        single { HibernateSessionManagerImpl(get(), get(), getAll()) } bind HibernateSessionManager::class
        single { DepartmentEntityMapperImpl() } bind DepartmentEntityMapper::class
        single { DepartmentUsersCountMapperImpl() } bind DepartmentUsersCountMapper::class

        single { DepartmentNameComponentImpl() } bind DepartmentNameComponent::class

        single { DepartmentServiceImpl(get(), get()) } bind DepartmentService::class
        single { DepartmentLinkerServiceImpl() } bind DepartmentLinkerService::class
        single { DepartmentUsersServiceImpl(get()) } bind DepartmentUsersService::class

        single { UpdateDepartmentDataValidator() } bind ModelValidator::class

        factory { params -> DepartmentRepositoryImpl(params.get()) } bind DepartmentRepository::class
        factory { params -> OrganizationRepositoryImpl(params.get()) } bind OrganizationRepository::class
        factory { params -> UserRepository(params.get()) } bind UserRepository::class
        factory { params -> DepartmentUserRepositoryImpl(params.get()) } bind DepartmentUserRepository::class
        factory { params -> DepartmentResponsibleUserRepositoryImpl(params.get()) } bind DepartmentResponsibleUserRepository::class
    }

    @BeforeEach
    fun beforeEach() {
        startKoin {
            modules(
                departmentModule
            )
        }
        transaction {
            getRepository<OrganizationRepository>().save(Organization(slug = "test", name = "Тест"))
        }
    }

    @AfterEach
    fun afterEach() {
        stopKoin()
    }

    @Test
    fun `list departments by pattern and language code`() {
        //Arrange
        val departmentService = get<DepartmentService>()
        (1..10).forEach {
            val data = DepartmentData.newBuilder()
                .setOrganizationId(1)
                .setParentDepartmentId("1655d06f-0cb6-41d0-82e7-88cfc189b2c5")
                .addAllNames(
                    listOf(
                        Departmentmanager.DepartmentName.newBuilder()
                            .setLanguageCode("en")
                            .setName("Test$it")
                            .build(),
                        Departmentmanager.DepartmentName.newBuilder()
                            .setLanguageCode("ru")
                            .setName("Тест$it")
                            .build(),
                    )
                )
                .build()
            departmentService.createDepartment(data, responsileUserIds = listOf(1))
        }
        //Act
        val departments = departmentService.getOrganizationDepartmentsByPattern(1,false,"st3", "en")

        //Assert
        assertEquals(1, departments.size)
        assertEquals("Тест3", departments[0].data.namesList.firstOrNull{it.languageCode == "ru"}!!.name)
    }
}
