package ru.yandex.crm.apphost.kotlin.handlers.entitymanager.api

import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.api.constant.FlagConstants
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.LinkedEntitiesManager
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.constant.EntityRecordField
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.mappers.AggregatorLinkMapper
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.mappers.EntityAttributeValueMapper
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.mappers.EntityMetaMapper
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.mappers.EntityRecordMapper
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.mappers.EntitySchemaMapper
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.mappers.ValidationResultMapper
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.model.LinkedEntityData
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.EntityValidator
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityMeta
import ru.yandex.crm.apphost.proto.BusinessEntityServiceBase
import ru.yandex.crm.apphost.proto.EntityRecordManagerProto
import ru.yandex.crm.apphost.proto.entitystorage.Entitystorage
import ru.yandex.crm.proto.gallifrey.entitymanager.Entitymanager
import ru.yandex.crm.proto.gallifrey.entitystorage.Entitystorage.EntityUser
import ru.yandex.web.apphost.api.request.RequestMeta
import java.util.UUID

class BusinessEntityService(
    private val entitySchemaMapper: EntitySchemaMapper,
    private val entityRecordMapper: EntityRecordMapper,
    private val entityValidator: EntityValidator,
    private val validationResultMapper: ValidationResultMapper,
    private val entityMetaMapper: EntityMetaMapper,
    private val linkedEntitiesManager: LinkedEntitiesManager,
    private val aggregatorLinkMapper: AggregatorLinkMapper,
    private val entityAttributeValueMapper: EntityAttributeValueMapper,
) : BusinessEntityServiceBase() {
    override fun create(
        meta: RequestMeta,
        request: EntityRecordManagerProto.CreateEntityRecordRequest
    ): EntityRecordManagerProto.CreateEntityRecordResponse {
        val entityMeta = entityMetaMapper.fromProtobufModel(request.entityMeta)
        val entityRecordData = request.entityRecordData
        val validationResult = entityValidator.validate(entityMeta, entityRecordData.data.attributesList)
        if (!validationResult.isSuccess) {
            return EntityRecordManagerProto.CreateEntityRecordResponse.newBuilder()
                .setValidationResult(validationResultMapper.toProtoValidationResult(validationResult))
                .build()
        }

        val entitySchemaProto = entitySchemaMapper.toProtoEntitySchema(entityMeta)
        val author = EntityUser.newBuilder()
            .setUserId(request.currentUser.userId)
            .setRole(EntityUser.Role.AUTHOR)
            .build()

        val entityRecord = entityRecordMapper.toEntityRecord(
            entityRecordData.data,
            entityMeta,
            entitySchemaProto,
            listOf(author),
            request.organizationId.value,
        )

        val response = EntityRecordManagerProto.CreateEntityRecordResponse.newBuilder()

        val createCommand = Entitystorage.CreateEntityCommand.newBuilder()
            .setSchema(entitySchemaProto)
            .setEntity(entityRecord)
            .setOrganizationId(request.organizationId.value)
            .build()
        response.createCommand = createCommand

        val linkedEntities =
            linkedEntitiesManager.getLinkedEntities(
                entityMeta,
                request.entityRecordData.data.attributesList,
                request.organizationId.value
            )
        if (linkedEntities.isNotEmpty()) {
            response.addAllLoadLinkedRecordsCommands(toGetByIdCommands(linkedEntities, request.organizationId.value))
        }

        return response
            .build()
    }

    override fun update(
        meta: RequestMeta,
        request: EntityRecordManagerProto.UpdateEntityRecordRequest
    ): EntityRecordManagerProto.UpdateEntityRecordResponse {
        val entityMeta = entityMetaMapper.fromProtobufModel(request.entityMeta)
        val entityRecordData = request.entityRecordData
        val validationResult = entityValidator.validate(entityMeta, entityRecordData.data.attributesList)
        if (!validationResult.isSuccess) {
            return EntityRecordManagerProto.UpdateEntityRecordResponse.newBuilder()
                .setValidationResult(validationResultMapper.toProtoValidationResult(validationResult))
                .build()
        }

        val entitySchemaProto = entitySchemaMapper.toProtoEntitySchema(entityMeta)
        val entityRecord = entityRecordMapper.toEntityRecord(
            entityRecordData.data,
            entityMeta,
            entitySchemaProto,
            listOf(),
            request.organizationId.value,
        )
        val response = EntityRecordManagerProto.UpdateEntityRecordResponse.newBuilder()

        val updatedEntityRecord = entityRecord.toBuilder()
            .setId(request.entityRecordId.value)
            .setVersion(entityRecordData.entityRecordVersion)
            .build()
        val updateCommand = Entitystorage.UpdateEntityCommand.newBuilder()
            .setSchema(entitySchemaProto)
            .setEntity(updatedEntityRecord)
            .setOrganizationId(request.organizationId.value)
            .build()
        response.updateCommand = updateCommand

        val linkedEntities =
            linkedEntitiesManager.getLinkedEntities(
                entityMeta,
                request.entityRecordData.data.attributesList,
                request.organizationId.value
            )
        if (linkedEntities.isNotEmpty()) {
            response.addAllLoadLinkedRecordsCommands(toGetByIdCommands(linkedEntities, request.organizationId.value))
        }

        return response
            .build()
    }

    override fun delete(
        meta: RequestMeta,
        request: EntityRecordManagerProto.DeleteEntityRecordRequest
    ): EntityRecordManagerProto.DeleteEntityRecordResponse {
        val entityMeta = entityMetaMapper.fromProtobufModel(request.entityMeta)
        val entitySchema = entitySchemaMapper.toProtoEntitySchema(entityMeta)
        val deleteCommand = Entitystorage.DeleteEntityCommand.newBuilder()
            .setSchema(entitySchema)
            .setOrganizationId(request.organizationId.value)
            .setId(request.entityRecord.id)
            .build()

        val response = EntityRecordManagerProto.DeleteEntityRecordResponse.newBuilder()
            .setDeleteCommand(deleteCommand)

        val linkedEntities =
            linkedEntitiesManager.getLinkedEntities(
                entityMeta,
                request.entityRecord.attributesList,
                request.organizationId.value
            )
        if (linkedEntities.isNotEmpty()) {
            response.addAllLoadLinkedRecordsCommands(toGetByIdCommands(linkedEntities, request.organizationId.value))
        }

        return response
            .build()
    }

    override fun get(
        meta: RequestMeta,
        request: EntityRecordManagerProto.GetEntityRecordsRequest
    ): EntityRecordManagerProto.GetEntityRecordsResponse {
        val entityMeta = entityMetaMapper.fromProtobufModel(request.entityMeta)
        val entitySchema = entitySchemaMapper.toProtoEntitySchema(entityMeta)
        val filters = entitySchemaMapper.toAttributeFilters(
            request.entityFilter.attributeFiltersList,
            entityMeta,
            entitySchema
        )

        val getListCommand = Entitystorage.ListEntitesCommand.newBuilder()
            .setSchema(entitySchema)
            .setOrganizationId(request.organizationId.value)
            .addAllFilters(filters)
            .build()

        val responseBuilder = EntityRecordManagerProto.GetEntityRecordsResponse.newBuilder()
            .setGetListCommand(getListCommand)

        val internalId = request.entityFilter.attributeFiltersList
            .firstOrNull { it.attributeShortName == EntityRecordField.INTERNAL_ID_FIELD_NAME }
            ?.expressionsList
            ?.first()
            ?.numberFilter?.longValue

        if (internalId != null) {
            val linkFilters = linkedEntitiesManager.getLinkFilters(entityMeta, internalId)
            if (linkFilters.isNotEmpty()) {
                responseBuilder.addAllLinkFilters(aggregatorLinkMapper.toFilterProtobufModel(linkFilters))
            }
        }

        return responseBuilder
            .build()
    }

    override fun compose(
        meta: RequestMeta,
        request: EntityRecordManagerProto.ComposeEntityRecordRequest
    ): EntityRecordManagerProto.ComposeEntityRecordResponse {
        val entitiesResponse = request.entitiesResponse
        val entityMeta = request.entityMeta

        val entityRecords = entityRecordMapper.toEntityRecords(entitiesResponse, entityMeta)

        val responseBuilder = EntityRecordManagerProto.ComposeEntityRecordResponse.newBuilder()
            .addAllEntityRecords(entityRecords)

        if (meta.checkFlag(FlagConstants.CREATE_RECORD)) {
            responseBuilder.createdEntityRecordId =
                Entitymanager.EntityRecordId.newBuilder()
                    .setValue(entitiesResponse.entitiesList.first().id)
                    .build()
        }

        return responseBuilder.build()
    }

    override fun createLinks(
        meta: RequestMeta,
        request: EntityRecordManagerProto.CreateEntityRecordLinksRequest
    ): EntityRecordManagerProto.CreateEntityRecordLinksResponse {
        val entityRecordId = UUID.fromString(request.entityRecordId.value)
        val entities = request.entityResponsesList.flatMap { it.entitiesList }

        val hierarchyConnections = linkedEntitiesManager.getHierarchyConnections(
            entityRecordId,
            entities,
            request.entityMeta.data.attributesList
        )

        linkedEntitiesManager.saveConnections(hierarchyConnections)

        val flatConnections = linkedEntitiesManager.generateFlatConnections(
            hierarchyConnections,
            entities,
            request.organizationId.value
        )

        val links = aggregatorLinkMapper.toLinkProtobufModel(flatConnections)

        return EntityRecordManagerProto.CreateEntityRecordLinksResponse.newBuilder()
            .addAllLinks(links)
            .build()
    }

    override fun deleteLinks(
        meta: RequestMeta,
        request: EntityRecordManagerProto.DeleteEntityRecordLinksRequest
    ): EntityRecordManagerProto.DeleteEntityRecordLinksResponse {
        val entityRecordId = UUID.fromString(request.entityRecordId.value)

        val hierarchConnections = linkedEntitiesManager.getHierarchyConnections(
            entityRecordId,
            request.entityResponsesList.flatMap { it.entitiesList },
            request.entityMeta.data.attributesList
        )

        linkedEntitiesManager.deleteConnections(hierarchConnections)

        val links = aggregatorLinkMapper.toLinkProtobufModel(hierarchConnections)

        return EntityRecordManagerProto.DeleteEntityRecordLinksResponse.newBuilder()
            .addAllLinks(links)
            .build()
    }

    private fun generateListWithIdFilterCommand(
        entityMeta: EntityMeta,
        organizationId: Long,
        entityRecordId: String
    ): Entitystorage.ListEntitesCommand {
        val entitySchema = entitySchemaMapper.toProtoEntitySchema(entityMeta)
        return Entitystorage.ListEntitesCommand.newBuilder()
            .setSchema(entitySchema)
            .setOrganizationId(organizationId)
            .addFilters(entityAttributeValueMapper.getIdFilter(entityRecordId))
            .build()
    }

    private fun toGetByIdCommands(
        linkedEntities: Collection<LinkedEntityData>,
        organizationId: Long
    ) = linkedEntities.map {
        generateListWithIdFilterCommand(it.entityMeta, organizationId, it.entityRecordId.toString())
    }
}
