package ru.yandex.crm.apphost.kotlin.handlers.entitymanager.api

import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.EntityMetaManager
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.EntityMetaVersionManager
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.mappers.EntityMetaMapper
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.externaldata.ExternalDataCommandProvider
import ru.yandex.crm.apphost.proto.EntityMetaManagerProto
import ru.yandex.crm.apphost.proto.EntityMetaServiceBase
import ru.yandex.web.apphost.api.request.RequestContextBase
import ru.yandex.web.apphost.api.request.RequestMeta
import java.util.UUID

class EntityMetaService(
    private val entityMetaManager: EntityMetaManager,
    private val entityMetaVersionManager: EntityMetaVersionManager,
    private val entityMetaMapper: EntityMetaMapper,
    private val externalDataCommandProvider: ExternalDataCommandProvider,
) : EntityMetaServiceBase() {

    override fun create(
        meta: RequestMeta,
        request: EntityMetaManagerProto.CreateEntityMetaRequest
    ): EntityMetaManagerProto.CreateEntityMetaResponse {
        val command = request.command
        val entityMeta = entityMetaMapper.fromProtobufModel(command.entityMetaData)

        val createdEntityMeta = entityMetaManager.create(entityMeta, command.organizationId, command.userId)

        val protobufEntityData = entityMetaMapper.toProtobufModel(createdEntityMeta)

        return EntityMetaManagerProto.CreateEntityMetaResponse.newBuilder()
            .setEntityMeta(protobufEntityData)
            .build()
    }

    override fun delete(
        meta: RequestMeta,
        request: EntityMetaManagerProto.DeleteEntityMetaRequest
    ): EntityMetaManagerProto.DeleteEntityMetaResponse {
        val command = request.command

        entityMetaManager.delete(command.organizationId, UUID.fromString(command.entityMetaId))

        return EntityMetaManagerProto.DeleteEntityMetaResponse.newBuilder()
            .build()
    }

    override fun getById(
        meta: RequestMeta,
        request: EntityMetaManagerProto.GetEntityMetaByIdRequest
    ): EntityMetaManagerProto.GetEntityMetaServiceResponse {
        val command = request.command

        val entityMeta = entityMetaManager.getById(command.organizationId, UUID.fromString(command.entityMetaId))
        val protobufEntityMeta = entityMetaMapper.toProtobufModel(entityMeta)

        return EntityMetaManagerProto.GetEntityMetaServiceResponse.newBuilder()
            .addEntityMetas(protobufEntityMeta)
            .build()
    }

    override fun getByNameAndVersion(
        meta: RequestMeta,
        request: EntityMetaManagerProto.GetEntityMetaByNameAndVersionRequest
    ): EntityMetaManagerProto.GetEntityMetaServiceResponse {
        val organizationId = request.organizationId.value
        val shortName = request.metaNameWithVersion.entityMetaShortName
        val versionNumber = with(request.metaNameWithVersion) {
            if (this.hasEntityMetaVersion()) this.entityMetaVersion else null
        }

        val entityMeta = entityMetaManager.getByShortNameAndVersion(organizationId, shortName, versionNumber)

        val externalCommand = externalDataCommandProvider.getCommand(entityMeta.id!!)

        if (externalCommand !== null && meta is RequestContextBase) {
            meta.addFlag(externalCommand.commandName)
        } else if (meta is RequestContextBase) {
            meta.addFlag("internal_storage")
        }

        return EntityMetaManagerProto.GetEntityMetaServiceResponse.newBuilder()
            .addEntityMetas(entityMetaMapper.toProtobufModel(entityMeta))
            .build()
    }

    override fun getList(
        meta: RequestMeta,
        request: EntityMetaManagerProto.GetEntityMetaListRequest
    ): EntityMetaManagerProto.GetEntityMetaServiceResponse {
        val command = request.command

        val entityMetas = entityMetaManager.get(command.organizationId)
        val protobufEntityMeta = entityMetas.map {
            entityMetaMapper.toProtobufModel(it)
        }.toMutableList()

        return EntityMetaManagerProto.GetEntityMetaServiceResponse.newBuilder()
            .addAllEntityMetas(protobufEntityMeta)
            .build()
    }

    override fun addAttributes(
        meta: RequestMeta,
        request: EntityMetaManagerProto.AddEntityMeatAttributesRequest
    ): EntityMetaManagerProto.AddEntityMeatAttributesResponse {
        val command = request.command

        val updatedEntityMeta = entityMetaManager.addAttributes(
            UUID.fromString(command.entityMetaId),
            command.organizationId,
            command.userId,
            command.attributes.attributesList
        )
        val protoEntityMeta = entityMetaMapper.toProtobufModel(updatedEntityMeta)

        return EntityMetaManagerProto.AddEntityMeatAttributesResponse.newBuilder()
            .setEntityMeta(protoEntityMeta)
            .build()
    }

    override fun activateDraft(
        meta: RequestMeta,
        request: EntityMetaManagerProto.ActivateCurrentDraftRequest
    ): EntityMetaManagerProto.ActivateDraftEntityMetaVersionResponse {
        val command = request.command

        val newActivatedVersion = entityMetaVersionManager.activateCurrentDraft(
            command.organizationId,
            UUID.fromString(command.entityMetaId),
            command.userId
        )
        val protobufEntityMetaVersion = entityMetaMapper.toProtobufModel(newActivatedVersion)

        return EntityMetaManagerProto.ActivateDraftEntityMetaVersionResponse.newBuilder()
            .setEntityMetaVersion(protobufEntityMetaVersion)
            .build()
    }
}
