package ru.yandex.crm.apphost.kotlin.handlers.entitymanager.api

import ru.yandex.crm.apphost.kotlin.common.extensions.getAppHostType
import ru.yandex.crm.apphost.kotlin.common.http.parser.HttpParserHandlerBase
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.api.constant.FlagConstants
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.constant.EntityRecordField
import ru.yandex.crm.apphost.proto.EntityMetaManagerProto
import ru.yandex.crm.proto.gallifrey.entitymanager.Entitymanager
import ru.yandex.crm.proto.gallifrey.organizationmanager.Organizationmanager
import ru.yandex.passport.tvmauth.TvmClient
import ru.yandex.web.apphost.api.request.RequestContext
import java.util.*

class HttpParserHandler(tvmClient: TvmClient) : HttpParserHandlerBase(tvmClient) {
    override val path: String = "/init/businessattributes"

    init {

        get("/organization/{organizationId}/entitymeta") {
            val organizationId = pathParameters["organizationId"]!!.toLong()
            val command = EntityMetaManagerProto.GetEntityMetaListCommand.newBuilder()
                .setOrganizationId(organizationId)
                .build()
            context.addProtobufItem(
                EntityMetaManagerProto.GetEntityMetaListCommand.getDescriptor().getAppHostType(),
                command
            )
        }

        get("/organization/{organizationId}/entitymeta/{entityMetaId}") {
            val organizationId = pathParameters["organizationId"]!!.toLong()
            val entityMetaId = pathParameters["entityMetaId"]
            val command = EntityMetaManagerProto.GetEntityMetaByIdCommand.newBuilder()
                .setOrganizationId(organizationId)
                .setEntityMetaId(entityMetaId)
                .build()
            context.addProtobufItem(
                EntityMetaManagerProto.GetEntityMetaByIdCommand.getDescriptor().getAppHostType(),
                command
            )
        }

        post("/organization/{organizationId}/entitymeta") {
            val organizationId = pathParameters["organizationId"]!!.toLong()
            val body = request.content
            val data = Entitymanager.EntityMetaData.parseFrom(body)
            val command = EntityMetaManagerProto.CreateEntityMetaCommand.newBuilder()
                .setOrganizationId(organizationId)
                .setEntityMetaData(data)
                .build()
            context.addProtobufItem(
                EntityMetaManagerProto.CreateEntityMetaCommand.getDescriptor().getAppHostType(),
                command
            )
        }

        delete("/organization/{organizationId}/entitymeta/{entityMetaId}") {
            val organizationId = pathParameters["organizationId"]!!.toLong()
            val entityMetaId = pathParameters["entityMetaId"]
            val command = EntityMetaManagerProto.DeleteEntityMetaCommand.newBuilder()
                .setOrganizationId(organizationId)
                .setEntityMetaId(entityMetaId)
                .build()
            context.addProtobufItem(
                EntityMetaManagerProto.DeleteEntityMetaCommand.getDescriptor().getAppHostType(),
                command
            )
        }

        put("/organization/{organizationId}/entitymeta/{entityMetaId}/attributes") {
            val organizationId = pathParameters["organizationId"]!!.toLong()
            val entityMetaId = pathParameters["entityMetaId"]
            val body = request.content
            val data = Entitymanager.EntityAttributesData.parseFrom(body)
            val command = EntityMetaManagerProto.AddEntityMeatAttributesCommand.newBuilder()
                .setOrganizationId(organizationId)
                .setEntityMetaId(entityMetaId)
                .setAttributes(data)
                .build()
            context.addProtobufItem(
                EntityMetaManagerProto.AddEntityMeatAttributesCommand.getDescriptor().getAppHostType(),
                command
            )
        }

        post("/organization/{organizationId}/entitymeta/{entityMetaId}/draft/activate") {
            val organizationId = pathParameters["organizationId"]!!.toLong()
            val entityMetaId = pathParameters["entityMetaId"]
            val command = EntityMetaManagerProto.ActivateCurrentDraftEntityMetaVersionCommand.newBuilder()
                .setOrganizationId(organizationId)
                .setEntityMetaId(entityMetaId)
                .build()
            context.addProtobufItem(
                EntityMetaManagerProto.ActivateCurrentDraftEntityMetaVersionCommand.getDescriptor().getAppHostType(),
                command
            )
        }

        put("/organization/{organizationId}/entitymeta/{entityMetaId}/name") {
            val organizationId = pathParameters["organizationId"]!!.toLong()
            val entityMetaId = pathParameters["entityMetaId"]
            val body = request.content
            val data = Entitymanager.EntityMetaNameData.parseFrom(body)
            val command = EntityMetaManagerProto.UpdateEntityMetaNameCommand.newBuilder()
                .setOrganizationId(organizationId)
                .setEntityMetaId(entityMetaId)
                .setName(data)
                .build()
            context.addProtobufItem(
                EntityMetaManagerProto.UpdateEntityMetaNameCommand.getDescriptor().getAppHostType(),
                command
            )
        }

        delete("/organization/{organizationId}/entitymeta/{entityMetaId}/name/{languageCode}") {
            val organizationId = pathParameters["organizationId"]!!.toLong()
            val entityMetaId = pathParameters["entityMetaId"]
            val languageCode = pathParameters["languageCode"]
            val command = EntityMetaManagerProto.DeleteEntityMetaNameCommand.newBuilder()
                .setOrganizationId(organizationId)
                .setEntityMetaId(entityMetaId)
                .setLanguageCode(languageCode)
                .build()
            context.addProtobufItem(
                EntityMetaManagerProto.DeleteEntityMetaNameCommand.getDescriptor().getAppHostType(), command
            )
        }

        put("/organization/{organizationId}/entitymeta/{entityMetaId}/attribute/{shortName}/name") {
            val organizationId = pathParameters["organizationId"]!!.toLong()
            val entityMetaId = pathParameters["entityMetaId"]
            val shortName = pathParameters["shortName"]
            val body = request.content
            val data = Entitymanager.EntityAttributeNameData.parseFrom(body)
            val command = EntityMetaManagerProto.UpdateEntityAttributeNameCommand.newBuilder()
                .setOrganizationId(organizationId)
                .setEntityMetaId(entityMetaId)
                .setAttributeShortName(shortName)
                .setName(data)
                .build()
            context.addProtobufItem(
                EntityMetaManagerProto.UpdateEntityAttributeNameCommand.getDescriptor().getAppHostType(), command
            )
        }

        delete("/organization/{organizationId}/entitymeta/{entityMetaId}/attribute/{shortName}/name/{languageCode}") {
            val organizationId = pathParameters["organizationId"]!!.toLong()
            val entityMetaId = pathParameters["entityMetaId"]
            val languageCode = pathParameters["languageCode"]
            val shortName = pathParameters["shortName"]
            val command = EntityMetaManagerProto.DeleteEntityAttributeNameCommand.newBuilder()
                .setOrganizationId(organizationId)
                .setEntityMetaId(entityMetaId)
                .setLanguageCode(languageCode)
                .setAttributeShortName(shortName)
                .build()
            context.addProtobufItem(
                EntityMetaManagerProto.DeleteEntityAttributeNameCommand.getDescriptor().getAppHostType(), command
            )
        }

        post("/organization/{organizationId}/entity/{entityMetaShortName}/{entityMetaVersion}") {
            val organizationId = pathParameters["organizationId"]!!.toLong()
            val entityMetaShortName = pathParameters["entityMetaShortName"]!!
            val entityMetaVersion = pathParameters["entityMetaVersion"]!!.toInt()
            val body = request.content

            context.addOrganizationId(organizationId)
            context.addGetMetaCommand(entityMetaShortName, entityMetaVersion)
            context.addProtobufItem(
                Entitymanager.CreateEntityRecordData.getDescriptor().getAppHostType(),
                Entitymanager.CreateEntityRecordData.parseFrom(body)
            )
            context.addFlag(FlagConstants.CREATE_RECORD)
        }

        put("/organization/{organizationId}/entity/{entityMetaShortName}/{entityMetaVersion}/{entityRecordId}") {
            val organizationId = pathParameters["organizationId"]!!.toLong()
            val entityMetaShortName = pathParameters["entityMetaShortName"]!!
            val entityMetaVersion = pathParameters["entityMetaVersion"]!!.toInt()
            val entityRecordId = UUID.fromString(pathParameters["entityRecordId"]!!)
            val body = request.content

            context.addOrganizationId(organizationId)
            context.addGetMetaCommand(entityMetaShortName, entityMetaVersion)
            context.addEntityRecordId(entityRecordId)
            context.addProtobufItem(
                Entitymanager.UpdateEntityRecordData.getDescriptor().getAppHostType(),
                Entitymanager.UpdateEntityRecordData.parseFrom(body)
            )
            context.addFlag(FlagConstants.UPDATE_RECORD)
        }

        delete("/organization/{organizationId}/entity/{entityMetaShortName}/{entityRecordId}") {
            val organizationId = pathParameters["organizationId"]!!.toLong()
            val entityMetaShortName = pathParameters["entityMetaShortName"]!!
            val entityRecordId = UUID.fromString(pathParameters["entityRecordId"]!!)

            context.addOrganizationId(organizationId)
            context.addGetMetaCommand(entityMetaShortName)
            context.addEntityRecordId(entityRecordId)
            context.addEntityRecordIdFilter(entityMetaShortName, entityRecordId)
            context.addFlag(FlagConstants.DELETE_RECORD)
        }

        get("/organization/{organizationId}/entity/{entityMetaShortName}") {
            val organizationId = pathParameters["organizationId"]!!.toLong()
            val entityMetaShortName = pathParameters["entityMetaShortName"]!!
            val body = request.content

            context.addOrganizationId(organizationId)
            context.addGetMetaCommand(entityMetaShortName)
            context.addProtobufItem(
                Entitymanager.EntityFilter.getDescriptor().getAppHostType(),
                Entitymanager.EntityFilter.parseFrom(body)
            )
            context.addFlag(FlagConstants.GET_RECORDS)
        }
    }

    private fun RequestContext.addOrganizationId(organizationId: Long) = this.addProtobufItem(
        Organizationmanager.OrganizationId.getDescriptor().getAppHostType(),
        Organizationmanager.OrganizationId.newBuilder().setValue(organizationId).build()
    )

    private fun RequestContext.addGetMetaCommand(metaShortName: String, version: Int? = null) = this.addProtobufItem(
        EntityMetaManagerProto.GetEntityMetaByNameAndVersionCommand.getDescriptor().getAppHostType(),
        EntityMetaManagerProto.GetEntityMetaByNameAndVersionCommand.newBuilder()
            .setEntityMetaShortName(metaShortName)
            .apply { if (version !== null) this.entityMetaVersion = version }
            .build()
    )

    private fun RequestContext.addEntityRecordId(entityRecordId: UUID) = this.addProtobufItem(
        Entitymanager.EntityRecordId.getDescriptor().getAppHostType(),
        Entitymanager.EntityRecordId.newBuilder().setValue(entityRecordId.toString()).build()
    )

    private fun RequestContext.addEntityRecordIdFilter(entityMetaShortName: String, entityRecordId: UUID) = this.addProtobufItem(
        Entitymanager.EntityFilter.getDescriptor().getAppHostType(),
        Entitymanager.EntityFilter.newBuilder()
            .setEntityMetaShortName(entityMetaShortName)
            .addAttributeFilters(Entitymanager.EntityAttributeFilter.newBuilder()
                .setAttributeShortName(EntityRecordField.ID_FIELD_NAME)
                .addExpressions(Entitymanager.FilterExpression.newBuilder()
                    .setStringFilter(Entitymanager.FilterExpression.StringFilter.newBuilder()
                        .setValue(entityRecordId.toString())
                        .setOperator(Entitymanager.FilterExpression.StringFilter.Operator.EQUAL))))
            .build()
    )
}
