package ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.impl

import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.EntityMetaManager
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.mappers.EntityMetaMapper
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.EntityMetaVersionValidator
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.EntityAttributeRepository
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.EntityMetaRepository
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityMeta
import ru.yandex.crm.library.kotlin.database.hibernate.getRepository
import ru.yandex.crm.library.kotlin.database.hibernate.transaction
import ru.yandex.crm.proto.gallifrey.entitymanager.Entitymanager
import java.time.Instant
import java.util.*

class EntityMetaManagerImpl(
    private val entityMetaMapper: EntityMetaMapper,
    private val entityMetaVersionValidator: EntityMetaVersionValidator,
) : EntityMetaManager {
    override fun create(entityMeta: EntityMeta, organizationId: Long, userId: Long): EntityMeta {
        val newEntityMeta = transaction {
            val entityMetaRepository = getRepository<EntityMetaRepository>()

            val creationAt = Instant.now()

            entityMeta.organizationId = organizationId
            entityMeta.createdBy = userId
            entityMeta.createdAt = creationAt
            entityMeta.isDeleted = false
            entityMeta.entityMetaVersions.forEach {
                it.createdAt = creationAt
                it.createdBy = userId
                entityMetaVersionValidator.validate(it)
            }

            entityMetaRepository.save(entityMeta)
        }

        return newEntityMeta
    }

    override fun get(organizationId: Long): Collection<EntityMeta> {
        val entityMetaList = transaction {
            val repository = getRepository<EntityMetaRepository>()
            repository.findByOrganizationId(organizationId)
        }

        return entityMetaList
    }

    override fun getById(organizationId: Long, entityMetaId: UUID): EntityMeta {
        val entityMeta = transaction {
            val repository = getRepository<EntityMetaRepository>()
            repository.findById(organizationId, entityMetaId, loadHierarchically = true)
        } ?: error("The entity meta with $entityMetaId id was not found")

        return entityMeta
    }

    override fun getByShortNameAndVersion(organizationId: Long, shortName: String, versionNumber: Int?): EntityMeta {
        val entityMeta = transaction {
            val repository = getRepository<EntityMetaRepository>()
            repository.findByShortNameAndVersion(organizationId, shortName, versionNumber, loadHierarchically = true)
        } ?: error("The entity meta with $shortName short name and $versionNumber version was not found")

        return entityMeta
    }

    override fun delete(organizationId: Long, entityMetaId: UUID) {
        transaction {
            val repository = getRepository<EntityMetaRepository>()
            val entityMeta = repository.findById(organizationId, entityMetaId, loadHierarchically = false)
                ?: error("The entity meta with $entityMetaId id was not found")
            entityMeta.isDeleted = true

            repository.save(entityMeta)
        }
    }

    override fun addAttributes(entityMetaId: UUID, organizationId: Long, userId: Long, attributes: Collection<Entitymanager.EntityAttribute>): EntityMeta {
        val entityMeta = transaction {
            val repository = getRepository<EntityMetaRepository>()
            val draftEntityMeta = repository.findDraftById(organizationId, entityMetaId)
                ?: error("The draft for entity meta with $entityMetaId id was not found")

            val draftVersion = draftEntityMeta.entityMetaVersions.first()
            val attributeRepository = getRepository<EntityAttributeRepository>()
            val maxFieldNumber = attributeRepository.findNextFieldNumber(draftEntityMeta.id!!)

            val versionAttributes = entityMetaMapper.fromProtobufModel(attributes, draftEntityMeta, draftVersion, maxFieldNumber)
            draftVersion.attributes.addAll(versionAttributes)
            draftVersion.changedAt = Instant.now()
            draftVersion.changedBy = userId

            entityMetaVersionValidator.validate(draftVersion)

            repository.save(draftEntityMeta)
        }

        return entityMeta
    }
}
