package ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.impl

import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.HierarchyConnectionManager
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.LinkedEntitiesManager
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.model.ComplexTypeEnum
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.model.EntityFlatConnection
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.model.FlatConnection
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.model.HierarchyConnection
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.model.HierarchyEntityData
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.model.LinkEntityFilter
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.model.LinkFilter
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.model.LinkedAttributeData
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.model.LinkedEntityData
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.EntityMetaRepository
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityMeta
import ru.yandex.crm.library.kotlin.database.hibernate.getRepository
import ru.yandex.crm.library.kotlin.database.hibernate.transaction
import ru.yandex.crm.proto.gallifrey.entitymanager.Entitymanager
import ru.yandex.crm.proto.gallifrey.entitystorage.Entitystorage
import java.util.UUID

class LinkedEntitiesManagerImpl(
    private val hierarchyConnectionManager: HierarchyConnectionManager,
) : LinkedEntitiesManager {
    override fun getLinkedEntities(
        entityMeta: EntityMeta,
        attributes: Collection<Entitymanager.EntityRecordAttribute>,
        organizationId: Long
    ): Collection<LinkedEntityData> {
        val linkedAttributes = entityMeta.getLinkedAttributes()
            .associateBy { it.attributeName }

        if (linkedAttributes.isEmpty()) {
            return emptyList()
        }

        attributes.forEach {
            val linkAttributeData = linkedAttributes[it.name]
            if (linkAttributeData != null) {
                linkAttributeData.entityRecordId = UUID.fromString(it.value)
            }
        }

        val childEntities = linkedAttributes.values
            .filter { it.isFirst && it.entityRecordId != null }
            .map { HierarchyEntityData(it.entityMetaId, it.entityRecordId!!) }
        val allChildEntities = hierarchyConnectionManager.getAllChilds(childEntities, organizationId)

        val parentEntities = linkedAttributes.values
            .filter { !it.isFirst && it.entityRecordId != null }
            .map { HierarchyEntityData(it.entityMetaId, it.entityRecordId!!) }
        val allParentEntities = hierarchyConnectionManager.getAllParents(parentEntities, organizationId)

        val allLinkedEntities = allChildEntities + allParentEntities

        val allMetaIds = linkedAttributes.values
            .map { it.entityMetaId } +
            allLinkedEntities.map { it.metaId }

        val entityMetas = transaction {
            val entityMetaRepository = getRepository<EntityMetaRepository>()
            entityMetaRepository.findByIds(entityMeta.organizationId!!, allMetaIds)
        }.associateBy { it.id }

        val result = mutableListOf<LinkedEntityData>()

        linkedAttributes.values.forEach {
            if (it.entityRecordId != null && entityMetas[it.entityMetaId] != null) {
                result.add(LinkedEntityData(it.entityRecordId!!, entityMetas[it.entityMetaId]!!))
            }
        }
        allLinkedEntities.forEach {
            if (entityMetas[it.metaId] != null) {
                result.add(LinkedEntityData(it.recordId, entityMetas[it.metaId]!!))
            }
        }

        return result
    }

    override fun getHierarchyConnections(
        modifiedEntityId: UUID,
        entities: Collection<Entitystorage.EntityRecord>,
        attributes: Collection<Entitymanager.EntityAttribute>
    ): Collection<HierarchyConnection> {
        val modifiedEntity =
            entities.first { it.id == modifiedEntityId.toString() }
        val entitiesMap = entities.associateBy { it.metaId }
        val linkedAttributes = attributes
            .filter { it.hasLinkParameter() }
        val result = mutableListOf<HierarchyConnection>()
        for (attribute in linkedAttributes) {
            val linkedEntity = entitiesMap[attribute.linkParameter.target]
            if (linkedEntity !== null) {
                if (attribute.linkParameter.isFirst) {
                    result.add(
                        HierarchyConnection(
                            modifiedEntity.organizationId,
                            UUID.fromString(modifiedEntity.metaId),
                            UUID.fromString(modifiedEntity.id),
                            modifiedEntity.internalId,
                            UUID.fromString(linkedEntity.metaId),
                            UUID.fromString(linkedEntity.id),
                            linkedEntity.internalId,
                            attribute.linkParameter.isMultiple,
                            attribute.linkParameter.isFirst
                        )
                    )
                } else {
                    result.add(
                        HierarchyConnection(
                            modifiedEntity.organizationId,
                            UUID.fromString(linkedEntity.metaId),
                            UUID.fromString(linkedEntity.id),
                            linkedEntity.internalId,
                            UUID.fromString(modifiedEntity.metaId),
                            UUID.fromString(modifiedEntity.id),
                            modifiedEntity.internalId, attribute.linkParameter.isMultiple,
                            attribute.linkParameter.isFirst
                        )
                    )
                }
            }
        }

        return result
    }

    override fun saveConnections(hierarchyConnections: Collection<HierarchyConnection>) {
        for (hierarchyConnection in hierarchyConnections) {
            if (!hierarchyConnection.isMultiple) {
                if (hierarchyConnection.modifiedEntityIsFirst) {
                    hierarchyConnectionManager.deleteOccurrences(
                        hierarchyConnection.organizationId,
                        hierarchyConnection.firstMetaId,
                        hierarchyConnection.secondMetaId,
                        firstEntityId = hierarchyConnection.firstEntityId
                    )
                } else {
                    hierarchyConnectionManager.deleteOccurrences(
                        hierarchyConnection.organizationId,
                        hierarchyConnection.firstMetaId,
                        hierarchyConnection.secondMetaId,
                        secondEntityId = hierarchyConnection.secondEntityId
                    )
                }
            }

            hierarchyConnectionManager.saveOrUpdate(hierarchyConnection)
        }
    }

    override fun deleteConnections(hierarchyConnections: Collection<HierarchyConnection>) {
        for (hierarchyConnection in hierarchyConnections) {
            hierarchyConnectionManager.delete(hierarchyConnection)
        }
    }

    override fun generateFlatConnections(
        hierarchyConnections: Collection<HierarchyConnection>,
        entities: Collection<Entitystorage.EntityRecord>,
        organizationId: Long
    ): Collection<FlatConnection> {
        if (hierarchyConnections.isEmpty()) {
            return emptyList()
        }
        val result = mutableSetOf<FlatConnection>()

        val parentEntities = hierarchyConnections
            .map { HierarchyEntityData(it.firstMetaId, it.firstEntityId) }

        val allParentLinks = hierarchyConnectionManager.getAllParents(parentEntities, organizationId)
        val allParentEntities = allParentLinks + parentEntities

        val childEntities = hierarchyConnections
            .map { HierarchyEntityData(it.secondMetaId, it.secondEntityId) }

        val allChildLinks = hierarchyConnectionManager.getAllChilds(childEntities, organizationId)
        val allChildEntities = allChildLinks + childEntities

        val entitiesMap = entities.associateBy { UUID.fromString(it.id) }

        for (parentEntity in allParentEntities) {
            for (childEntity in allChildEntities) {
                if (parentEntity != childEntity && entitiesMap[parentEntity.recordId] != null && entitiesMap[childEntity.recordId] != null) {
                    result.add(
                        EntityFlatConnection(
                            organizationId,
                            parentEntity.metaId,
                            entitiesMap[parentEntity.recordId]!!.internalId,
                            childEntity.metaId,
                            entitiesMap[childEntity.recordId]!!.internalId
                        )
                    )
                }
            }
        }

        return result
    }

    override fun getLinkFilters(entityMeta: EntityMeta, internalRecordId: Long): Collection<LinkFilter> {
        val linkedAttributes = entityMeta.getLinkedAttributes()
        if (linkedAttributes.isEmpty()) {
            return emptyList()
        }

        val result = mutableListOf<LinkFilter>()
        for (linkAttribute in linkedAttributes) {
            val linkEntityFilter = LinkEntityFilter(
                entityMeta.id.toString(),
                internalRecordId,
                linkAttribute.entityMetaId.toString()
            )
            if (linkAttribute.isFirst) {
                result.add(LinkFilter(first = linkEntityFilter))
            } else {
                result.add(LinkFilter(second = linkEntityFilter))
            }
        }

        return result
    }

    private fun EntityMeta.getLinkedAttributes() =
        entityMetaVersions.first().attributes
            .filter { it.id!!.entityAttribute.complexType.name == ComplexTypeEnum.LINK.name }
            .map {
                LinkedAttributeData(
                    it.shortName!!,
                    it.id!!.entityAttribute.linkParameter!!.target!!,
                    it.id!!.entityAttribute.linkParameter!!.isFirst!!
                )
            }
}
