package ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.jooq

import org.jooq.Configuration
import org.jooq.Record2
import org.jooq.Table
import org.jooq.impl.DSL.field
import org.jooq.impl.DSL.name
import org.jooq.impl.DSL.row
import org.jooq.impl.DSL.select
import org.jooq.impl.DSL.values
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.HierarchyConnectionManager
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.model.HierarchyConnection
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.model.HierarchyEntityData
import ru.yandex.crm.generated.database.entitymanager.entitymanager.tables.references.ENTITY_HIERARCHY_CONNECTION
import java.util.*

class JooqHierarchyConnectionManager(
    private val jooqConfiguration: Configuration
) : HierarchyConnectionManager {

    override fun saveOrUpdate(hierarchyConnectedEntity: HierarchyConnection) {
        jooqConfiguration.dsl().transaction { cfg ->
            cfg.dsl()
                .insertInto(
                    ENTITY_HIERARCHY_CONNECTION,
                    ENTITY_HIERARCHY_CONNECTION.ORGANIZATION_ID,
                    ENTITY_HIERARCHY_CONNECTION.FIRST_META_ID,
                    ENTITY_HIERARCHY_CONNECTION.FIRST_RECORD_ID,
                    ENTITY_HIERARCHY_CONNECTION.SECOND_META_ID,
                    ENTITY_HIERARCHY_CONNECTION.SECOND_RECORD_ID
                )
                .values(
                    hierarchyConnectedEntity.organizationId,
                    hierarchyConnectedEntity.firstMetaId,
                    hierarchyConnectedEntity.firstEntityId,
                    hierarchyConnectedEntity.secondMetaId,
                    hierarchyConnectedEntity.secondEntityId
                )
                .onDuplicateKeyIgnore()
                .execute()
        }
    }

    override fun delete(hierarchyConnectedEntity: HierarchyConnection) {
        deleteOccurrences(
            hierarchyConnectedEntity.organizationId,
            hierarchyConnectedEntity.firstMetaId,
            hierarchyConnectedEntity.secondMetaId,
            hierarchyConnectedEntity.firstEntityId,
            hierarchyConnectedEntity.secondEntityId
        )
    }

    override fun deleteOccurrences(
        organizationId: Long,
        firstMetaId: UUID,
        secondMetaId: UUID,
        firstEntityId: UUID?,
        secondEntityId: UUID?
    ) {
        jooqConfiguration.dsl().transaction { cfg ->
            var command = cfg.dsl()
                .delete(ENTITY_HIERARCHY_CONNECTION)
                .where(ENTITY_HIERARCHY_CONNECTION.ORGANIZATION_ID.eq(organizationId))
                .and(ENTITY_HIERARCHY_CONNECTION.FIRST_META_ID.eq(firstMetaId))
                .and(ENTITY_HIERARCHY_CONNECTION.SECOND_META_ID.eq(secondMetaId))

            if (firstEntityId != null) {
                command = command
                    .and(ENTITY_HIERARCHY_CONNECTION.FIRST_RECORD_ID.eq(firstEntityId))
            }

            if (secondEntityId != null) {
                command = command
                    .and(ENTITY_HIERARCHY_CONNECTION.SECOND_RECORD_ID.eq(secondEntityId))
            }

            command.execute()
        }
    }

    override fun getAllParents(hierarchyEntities: Collection<HierarchyEntityData>, organizationId: Long): Collection<HierarchyEntityData> {
        if (hierarchyEntities.isEmpty()) {
            return emptyList()
        }

        val hierarchyEntityValues = getHierarchyEntityValues(hierarchyEntities)
        val hierarchyTable = name(HIERARCHY_TABLE_NAME).`as`(
            select(ENTITY_HIERARCHY_CONNECTION.FIRST_META_ID, ENTITY_HIERARCHY_CONNECTION.FIRST_RECORD_ID)
                .from(ENTITY_HIERARCHY_CONNECTION)
                .join(hierarchyEntityValues)
                .on(ENTITY_HIERARCHY_CONNECTION.SECOND_META_ID.eq(ENTITY_META_ID_FIELD))
                .and(ENTITY_HIERARCHY_CONNECTION.SECOND_RECORD_ID.eq(ENTITY_RECORD_ID_FIELD))
                .where(ENTITY_HIERARCHY_CONNECTION.ORGANIZATION_ID.eq(organizationId))
                .unionAll(
                    select(ENTITY_HIERARCHY_CONNECTION.FIRST_META_ID, ENTITY_HIERARCHY_CONNECTION.FIRST_RECORD_ID)
                        .from(ENTITY_HIERARCHY_CONNECTION)
                        .join(HIERARCHY_TABLE_NAME)
                        .on(
                            HIERARCHY_FIRST_META_ID.eq(
                                ENTITY_HIERARCHY_CONNECTION.SECOND_META_ID
                            )
                        )
                        .and(
                            HIERARCHY_FIRST_RECORD_ID.eq(
                                ENTITY_HIERARCHY_CONNECTION.SECOND_RECORD_ID
                            )
                        )
                        .where(ENTITY_HIERARCHY_CONNECTION.ORGANIZATION_ID.eq(organizationId))
                )
        )
        return jooqConfiguration.dsl().withRecursive(hierarchyTable)
            .select(HIERARCHY_FIRST_META_ID.`as`("metaId"), HIERARCHY_FIRST_RECORD_ID.`as`("recordId"))
            .from(hierarchyTable)
            .fetchInto(HierarchyEntityData::class.java)
    }

    override fun getAllChilds(hierarchyEntities: Collection<HierarchyEntityData>, organizationId: Long): Collection<HierarchyEntityData> {
        if (hierarchyEntities.isEmpty()) {
            return emptyList()
        }

        val hierarchyEntityValues = getHierarchyEntityValues(hierarchyEntities)
        val hierarchyTable = name(HIERARCHY_TABLE_NAME).`as`(
            select(ENTITY_HIERARCHY_CONNECTION.SECOND_META_ID, ENTITY_HIERARCHY_CONNECTION.SECOND_RECORD_ID)
                .from(ENTITY_HIERARCHY_CONNECTION)
                .join(hierarchyEntityValues)
                .on(ENTITY_HIERARCHY_CONNECTION.FIRST_META_ID.eq(ENTITY_META_ID_FIELD))
                .and(ENTITY_HIERARCHY_CONNECTION.FIRST_RECORD_ID.eq(ENTITY_RECORD_ID_FIELD))
                .where(ENTITY_HIERARCHY_CONNECTION.ORGANIZATION_ID.eq(organizationId))
                .unionAll(
                    select(ENTITY_HIERARCHY_CONNECTION.SECOND_META_ID, ENTITY_HIERARCHY_CONNECTION.SECOND_RECORD_ID)
                        .from(ENTITY_HIERARCHY_CONNECTION)
                        .join(HIERARCHY_TABLE_NAME)
                        .on(
                            HIERARCHY_SECOND_META_ID.eq(
                                ENTITY_HIERARCHY_CONNECTION.FIRST_META_ID
                            )
                        )
                        .and(
                            HIERARCHY_SECOND_RECORD_ID.eq(
                                ENTITY_HIERARCHY_CONNECTION.FIRST_RECORD_ID
                            )
                        )
                        .where(ENTITY_HIERARCHY_CONNECTION.ORGANIZATION_ID.eq(organizationId))
                )
        )
        return jooqConfiguration.dsl().withRecursive(hierarchyTable)
            .select(HIERARCHY_SECOND_META_ID.`as`("metaId"), HIERARCHY_SECOND_RECORD_ID.`as`("recordId"))
            .from(hierarchyTable)
            .fetchInto(HierarchyEntityData::class.java)
    }

    private fun getHierarchyEntityValues(hierarchyEntities: Collection<HierarchyEntityData>)
        : Table<Record2<UUID, UUID>> {
        val rows = hierarchyEntities.map {
            row(it.metaId, it.recordId)
        }.toTypedArray()
        return values(*rows)
            .`as`(ENTITY_DATA, ENTITY_DATA_META_ID, ENTITY_DATA_RECORD_ID)
    }

    companion object {
        const val HIERARCHY_TABLE_NAME = "hierarchy"
        const val ENTITY_DATA = "entity_data"
        const val ENTITY_DATA_META_ID = "meta_id"
        const val ENTITY_DATA_RECORD_ID = "record_id"

        val HIERARCHY_FIRST_META_ID = field(name(HIERARCHY_TABLE_NAME, ENTITY_HIERARCHY_CONNECTION.FIRST_META_ID.name))
        val HIERARCHY_FIRST_RECORD_ID = field(name(HIERARCHY_TABLE_NAME, ENTITY_HIERARCHY_CONNECTION.FIRST_RECORD_ID.name))
        val HIERARCHY_SECOND_META_ID = field(name(HIERARCHY_TABLE_NAME, ENTITY_HIERARCHY_CONNECTION.SECOND_META_ID.name))
        val HIERARCHY_SECOND_RECORD_ID = field(name(HIERARCHY_TABLE_NAME, ENTITY_HIERARCHY_CONNECTION.SECOND_RECORD_ID.name))

        val ENTITY_META_ID_FIELD = field(name(ENTITY_DATA, ENTITY_DATA_META_ID), UUID::class.java)
        val ENTITY_RECORD_ID_FIELD = field(name(ENTITY_DATA, ENTITY_DATA_RECORD_ID), UUID::class.java)
    }
}
