package ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.mappers.impl

import ru.yandex.crm.apphost.kotlin.common.extensions.toProtobufTimestamp
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.constant.EntityRecordField
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.mappers.AttributeValueProvider
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.mappers.EntityAttributeValueMapper
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.mappers.extension.toBooleanValue
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.mappers.extension.toInstantValue
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.mappers.extension.toIntValue
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.mappers.extension.toUuidValue
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.model.SimpleTypeEnum
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityAttribute
import ru.yandex.crm.proto.gallifrey.entitymanager.Entitymanager
import ru.yandex.crm.proto.gallifrey.entitystorage.Entitystorage
import ru.yandex.crm.proto.gallifrey.entitystorage.Entitystorage.EqualityFilter
import ru.yandex.crm.proto.gallifrey.entitystorage.Entitystorage.NumberRangeFilter

class EntityAttributeValueMapperImpl(
    private val valueProvider: AttributeValueProvider
) : EntityAttributeValueMapper {
    override fun toProtoAttributeValue(metaAttribute: EntityAttribute, value: String): Entitystorage.AttributeValue {
        val builder = Entitystorage.AttributeValue.newBuilder()

        when (metaAttribute.complexType.simpleType.toEnum()) {
            SimpleTypeEnum.INT -> {
                val intValue = metaAttribute.toIntValue(value)
                if (intValue != null) {
                    builder.intValue = intValue
                }
            }

            SimpleTypeEnum.UUID -> {
                val uuidValue = metaAttribute.toUuidValue(value)
                if (uuidValue != null) {
                    builder.uuidValue = uuidValue.toString()
                }
            }

            SimpleTypeEnum.TIMESTAMP -> {
                val instantValue = metaAttribute.toInstantValue(value)
                if (instantValue != null) {
                    builder.timestampValue = instantValue.toProtobufTimestamp()
                }
            }

            SimpleTypeEnum.BOOL -> {
                builder.boolValue = metaAttribute.toBooleanValue(value)
            }

            else -> {
                val stringValue = valueProvider.getValue(metaAttribute, value)
                if (stringValue != null) {
                    builder.stringValue = stringValue.toString()
                }
            }
        }

        return builder.build()
    }

    override fun toProtoAttributeFilter(
        metaAttribute: EntityAttribute,
        filter: Entitymanager.FilterExpression,
        attributeSchema: Entitystorage.AttributeSchema,
    ): Entitystorage.AttributeFilter {
        val builder = Entitystorage.AttributeFilter.newBuilder()
        val attributeSimpleType = metaAttribute.complexType.simpleType.toEnum()

        when (filter.filterCase) {
            Entitymanager.FilterExpression.FilterCase.NUMBER_FILTER -> builder.addNumberFilter(
                filter.numberFilter,
                attributeSimpleType
            )

            Entitymanager.FilterExpression.FilterCase.STRING_FILTER -> builder.addStringFilter(
                filter.stringFilter,
                attributeSimpleType
            )

            Entitymanager.FilterExpression.FilterCase.TIMESTAMP_FILTER -> builder.addTimestampFilter(
                filter.timestampFilter,
                attributeSimpleType
            )

            Entitymanager.FilterExpression.FilterCase.LINKED_ENTITIES_ATTRIBUTES_FILTER -> TODO("should it be here?")
            Entitymanager.FilterExpression.FilterCase.FILTER_NOT_SET -> error("empty filter")
        }

        return builder.setSchema(attributeSchema).build()
    }

    override fun getIdFilter(entityRecordId: String): Entitystorage.AttributeFilter {
        return Entitystorage.AttributeFilter.newBuilder()
            .setSchema(
                Entitystorage.AttributeSchema.newBuilder()
                    .setName(EntityRecordField.ID_FIELD_NAME)
                    .setType(Entitystorage.AttributeSchema.AttributeType.UUID)
            )
            .setEquality(
                EqualityFilter.newBuilder()
                    .setValue(
                        Entitystorage.AttributeValue.newBuilder()
                            .setUuidValue(entityRecordId)
                            .build()
                    )
                    .build()
            )
            .build()
    }

    override fun fromProtoAttributeValue(attributeValue: Entitystorage.AttributeValue): String? {
        if (attributeValue.hasBoolValue()) {
            return attributeValue.boolValue.toString()
        }

        if (attributeValue.hasIntValue()) {
            return attributeValue.intValue.toString()
        }

        if (attributeValue.hasStringValue()) {
            return attributeValue.stringValue
        }

        if (attributeValue.hasTimestampValue()) {
            return attributeValue.timestampValue.toString()
        }

        if (attributeValue.hasUuidValue()) {
            return attributeValue.uuidValue
        }

        return null
    }
}

private fun Entitystorage.AttributeFilter.Builder.addNumberFilter(
    numberFilter: Entitymanager.FilterExpression.NumberFilter,
    attributeSimpleType: SimpleTypeEnum
) {
    fun buildEquality(): EqualityFilter.Builder {
        return EqualityFilter.newBuilder()
            .setValue(Entitystorage.AttributeValue.newBuilder()
                .setIntValue(numberFilter.intValue))
    }

    fun buildLessRange(): NumberRangeFilter.Builder {
        return NumberRangeFilter.newBuilder()
            .setMaxValue(numberFilter.intValue.toDouble())
    }

    fun buildMoreRange(): NumberRangeFilter.Builder {
        return NumberRangeFilter.newBuilder()
            .setMinValue(numberFilter.intValue.toDouble())
    }

    if (attributeSimpleType !== SimpleTypeEnum.INT) {
        error("types mismatch")
    }

    if (numberFilter.valueCase != Entitymanager.FilterExpression.NumberFilter.ValueCase.INT_VALUE) {
        error("unrecognized field type")
    }

    when (numberFilter.operator) {
        Entitymanager.FilterExpression.NumberFilter.Operator.EQUAL -> this.equality = buildEquality().build()
        Entitymanager.FilterExpression.NumberFilter.Operator.NOT_EQUAL -> TODO("not supported")
        Entitymanager.FilterExpression.NumberFilter.Operator.LESS -> this.numberRange = buildLessRange().build()
        Entitymanager.FilterExpression.NumberFilter.Operator.LESS_OR_EQUAL -> this.numberRange = buildLessRange().setIncludeMax(true).build()
        Entitymanager.FilterExpression.NumberFilter.Operator.MORE -> this.numberRange = buildMoreRange().build()
        Entitymanager.FilterExpression.NumberFilter.Operator.MORE_OR_EQUAL -> this.numberRange = buildMoreRange().setIncludeMin(true).build()
        Entitymanager.FilterExpression.NumberFilter.Operator.UNRECOGNIZED -> error("unsupported filter operator")
    }
}

private fun Entitystorage.AttributeFilter.Builder.addStringFilter(
    stringFilter: Entitymanager.FilterExpression.StringFilter,
    attributeSimpleType: SimpleTypeEnum
) {
    if (attributeSimpleType !== SimpleTypeEnum.STRING) {
        error("types mismatch")
    }

    if (stringFilter.operator != Entitymanager.FilterExpression.StringFilter.Operator.EQUAL) {
        error("unsupported filter operator")
    }

    this.equalityBuilder.valueBuilder.stringValue = stringFilter.value
}

private fun Entitystorage.AttributeFilter.Builder.addTimestampFilter(
    timestampFilter: Entitymanager.FilterExpression.TimestampFilter,
    attributeSimpleType: SimpleTypeEnum
) {
    if (attributeSimpleType !== SimpleTypeEnum.TIMESTAMP) {
        error("types mismatch")
    }

    if (timestampFilter.operator != Entitymanager.FilterExpression.TimestampFilter.Operator.EQUAL) {
        error("unsupported filter operator")
    }

    this.equalityBuilder.valueBuilder.timestampValue = timestampFilter.value
}
