package ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.mappers.impl

import ru.yandex.crm.apphost.kotlin.common.extensions.*
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.ComplexTypeManager
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.mappers.EntityMetaMapper
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.*
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.parameter.*
import ru.yandex.crm.proto.gallifrey.entitymanager.Entitymanager
import java.util.UUID

class EntityMetaMapperImpl(
    private val complexTypeManager: ComplexTypeManager
) : EntityMetaMapper {
    override fun fromProtobufModel(entityMeta: Entitymanager.EntityMeta): EntityMeta {
        val meta = fromProtobufModel(entityMeta.data)
        meta.id = UUID.fromString(entityMeta.id)
        meta.organizationId = entityMeta.organizationId

        return meta
    }

    override fun fromProtobufModel(entityMetaData: Entitymanager.EntityMetaData): EntityMeta {
        val entityMeta = EntityMeta(
            shortName = entityMetaData.shortName
        ).apply {
            names = entityMetaData.namesList.map {
                EntityMetaName(EntityMetaNameId(this, it.languageCode), it.name)
            }.toMutableSet()
        }

        val entityMetaVersion = EntityMetaVersion(
            entityMeta = entityMeta,
            versionNumber = INIT_VERSION_NUMBER,
            status = EntityVersionStatus(EntityVersionStatusEnum.DRAFT.value)
        )

        val versionAttributes = fromProtobufModel(entityMetaData.attributesList, entityMeta, entityMetaVersion, 0)

        entityMetaVersion.apply {
            this.attributes = versionAttributes
        }

        entityMeta.apply {
            entityMetaVersions = mutableSetOf(entityMetaVersion)
        }

        return entityMeta
    }

    override fun fromProtobufModel(
        attributes: Collection<Entitymanager.EntityAttribute>,
        entityMeta: EntityMeta,
        entityMetaVersion: EntityMetaVersion,
        fieldNumberCounter: Int)
    : MutableSet<EntityMetaVersionAttribute> {
        var counter = fieldNumberCounter
        return attributes.map {
            // TODO: костыль с fieldNumber, пока оно не уехало из меты
            val attribute = fromProtobufModel(it, entityMeta, if (it.hasFieldNumber()) it.fieldNumber else counter++)

            EntityMetaVersionAttribute(
                EntityMetaVersionAttributeId(entityMetaVersion, attribute),
                it.shortName
            )
        }.toMutableSet()
    }

    override fun toProtobufModel(entityMeta: EntityMeta): Entitymanager.EntityMeta {
        return Entitymanager.EntityMeta.newBuilder()
            .setId(entityMeta.id.toString())
            .setOrganizationId(entityMeta.organizationId!!)
            .setData(toProtobufModelData(entityMeta))
            .build()
    }

    override fun toProtobufModel(entityMetaVersion: EntityMetaVersion): Entitymanager.EntityMetaVersion {
        return Entitymanager.EntityMetaVersion.newBuilder()
            .setEntityMetaId(entityMetaVersion.entityMeta.id.toString())
            .setVersionNumber(entityMetaVersion.versionNumber!!)
            .setStatus(Entitymanager.EntityMetaVersion.Status.forNumber(entityMetaVersion.status.id!!.toInt()))
            .build()
    }

    private fun toProtobufModelData(entityMeta: EntityMeta): Entitymanager.EntityMetaData {
        return Entitymanager.EntityMetaData.newBuilder()
            .setShortName(entityMeta.shortName)
            .addAllNames(entityMeta.names.map { toProtobufModel(it) })
            .addAllAttributes(entityMeta.entityMetaVersions.first().attributes.map {
                toProtobufModel(
                    it.id!!.entityAttribute,
                    it.shortName!!
                )
            })
            .build()
    }

    private fun toProtobufModel(entityMetaName: EntityMetaName): Entitymanager.EntityMetaNameData {
        return Entitymanager.EntityMetaNameData.newBuilder()
            .setName(entityMetaName.name)
            .setLanguageCode(entityMetaName.id!!.languageCode)
            .build()
    }

    private fun toProtobufModel(
        entityAttribute: EntityAttribute,
        shortName: String
    ): Entitymanager.EntityAttribute {
        val builder = Entitymanager.EntityAttribute.newBuilder()
            .setShortName(shortName)
            .setIsNullable(entityAttribute.isNullable!!)
            .setIsRequired(entityAttribute.isRequired!!)
            .addAllNames(entityAttribute.names.map { toProtobufModel(it) })
            .setType(Entitymanager.EntityAttribute.Type.valueOf(entityAttribute.complexType.name!!))
            .setFieldNumber(entityAttribute.fieldNumber!!)

        if (entityAttribute.dateParameter != null) {
            builder.setDateParameter(toProtobufModel(entityAttribute.dateParameter!!))
        }

        if (entityAttribute.timeParameter != null) {
            builder.setTimeParameter(toProtobufModel(entityAttribute.timeParameter!!))
        }

        if (entityAttribute.timestampParameter != null) {
            builder.setTimestampParameter(toProtobufModel(entityAttribute.timestampParameter!!))
        }

        if (entityAttribute.doubleParameter != null) {
            builder.setDoubleParameter(toProtobufModel(entityAttribute.doubleParameter!!))
        }

        if (entityAttribute.integerParameter != null) {
            builder.setIntegerParameter(toProtobufModel(entityAttribute.integerParameter!!))
        }

        if (entityAttribute.stringParameter != null) {
            builder.setStringParameter(toProtobufModel(entityAttribute.stringParameter!!))
        }

        if (entityAttribute.linkParameter != null) {
            builder.setLinkParameter(toProtobufModel(entityAttribute.linkParameter!!))
        }

        return builder.build()
    }

//region Entities to Protobuf Attribute Parameters mappers

    private fun toProtobufModel(attributeParameter: EntityDateAttributeParameter): Entitymanager.EntityDateAttributeParameter {
        val builder = Entitymanager.EntityDateAttributeParameter.newBuilder()
            .setTodayIsDefault(attributeParameter.todayIsDefault!!)

        if (attributeParameter.defaultValue != null) {
            builder.setDefaultValue(attributeParameter.defaultValue!!.toProtobufDate())
        }

        return builder.build()
    }

    private fun toProtobufModel(attributeParameter: EntityTimeAttributeParameter): Entitymanager.EntityTimeAttributeParameter {
        val builder = Entitymanager.EntityTimeAttributeParameter.newBuilder()
            .setNowIsDefault(attributeParameter.nowIsDefault!!)

        if (attributeParameter.defaultValue != null) {
            builder.setDefaultValue(attributeParameter.defaultValue!!.toProtobufTimestamp())
        }

        return builder.build()
    }

    private fun toProtobufModel(attributeParameter: EntityTimestampAttributeParameter): Entitymanager.EntityTimestampAttributeParameter {
        val builder = Entitymanager.EntityTimestampAttributeParameter.newBuilder()
            .setNowIsDefault(attributeParameter.nowIsDefault!!)

        if (attributeParameter.defaultValue != null) {
            builder.setDefaultValue(attributeParameter.defaultValue!!.toProtobufTimestamp())
        }

        return builder.build()
    }

    private fun toProtobufModel(attributeParameter: EntityIntegerAttributeParameter): Entitymanager.EntityIntegerAttributeParameter {
        val builder = Entitymanager.EntityIntegerAttributeParameter.newBuilder()
        if (attributeParameter.defaultValue != null) {
            builder.setDefaultValue(attributeParameter.defaultValue!!)
        }

        if (attributeParameter.minValue != null) {
            builder.setMinValue(attributeParameter.minValue!!)
        }

        if (attributeParameter.maxValue != null) {
            builder.setMaxValue(attributeParameter.maxValue!!)
        }

        return builder.build()
    }

    private fun toProtobufModel(attributeParameter: EntityDoubleAttributeParameter): Entitymanager.EntityDoubleAttributeParameter {
        val builder = Entitymanager.EntityDoubleAttributeParameter.newBuilder()
        if (attributeParameter.defaultValue != null) {
            builder.setDefaultValue(attributeParameter.defaultValue!!)
        }

        if (attributeParameter.minValue != null) {
            builder.setMinValue(attributeParameter.minValue!!)
        }

        if (attributeParameter.maxValue != null) {
            builder.setMaxValue(attributeParameter.maxValue!!)
        }

        return builder.build()
    }

    private fun toProtobufModel(attributeParameter: EntityStringAttributeParameter): Entitymanager.EntityStringAttributeParameter {
        val builder = Entitymanager.EntityStringAttributeParameter.newBuilder()
        if (attributeParameter.defaultValue != null) {
            builder.setDefaultValue(attributeParameter.defaultValue!!)
        }

        if (attributeParameter.maxLength != null) {
            builder.setMaxLength(attributeParameter.maxLength!!)
        }

        return builder.build()
    }

    private fun toProtobufModel(attributeParameter: EntityLinkAttributeParameter): Entitymanager.EntityLinkAttributeParameter {
        val builder = Entitymanager.EntityLinkAttributeParameter.newBuilder()
            .setIsFirst(attributeParameter.isFirst!!)
            .setIsMultiple(attributeParameter.isMultiple!!)
            .setTarget(attributeParameter.target.toString())

        return builder.build()
    }
//endregion

    private fun toProtobufModel(entityAttributeName: EntityAttributeName): Entitymanager.EntityAttributeNameData {
        return Entitymanager.EntityAttributeNameData.newBuilder()
            .setName(entityAttributeName.name)
            .setLanguageCode(entityAttributeName.id!!.languageCode)
            .build()
    }

    private fun fromProtobufModel(
        attribute: Entitymanager.EntityAttribute,
        entityMeta: EntityMeta,
        fieldNumber: Int
    ): EntityAttribute {
        val entityAttribute = EntityAttribute(
            entityMeta = entityMeta,
            fieldNumber = fieldNumber
        ).apply {
            complexType = complexTypeManager.getComplexTypeByName(attribute.type.name)
            isNullable = attribute.isNullable
            isRequired = attribute.isRequired
            names = attribute.namesList.map {
                EntityAttributeName(EntityAttributeNameId(this, it.languageCode), it.name)
            }.toMutableSet()
        }

        entityMeta.entityAttributes.add(entityAttribute)

        if (attribute.hasDateParameter()) {
            entityAttribute.dateParameter = fromProtobufModel(entityAttribute, attribute.dateParameter)
        }

        if (attribute.hasTimeParameter()) {
            entityAttribute.timeParameter = fromProtobufModel(entityAttribute, attribute.timeParameter)
        }

        if (attribute.hasTimestampParameter()) {
            entityAttribute.timestampParameter = fromProtobufModel(entityAttribute, attribute.timestampParameter)
        }

        if (attribute.hasDoubleParameter()) {
            entityAttribute.doubleParameter = fromProtobufModel(entityAttribute, attribute.doubleParameter)
        }

        if (attribute.hasIntegerParameter()) {
            entityAttribute.integerParameter = fromProtobufModel(entityAttribute, attribute.integerParameter)
        }

        if (attribute.hasStringParameter()) {
            entityAttribute.stringParameter = fromProtobufModel(entityAttribute, attribute.stringParameter)
        }

        if (attribute.hasLinkParameter()) {
            entityAttribute.linkParameter = fromProtobufModel(entityAttribute, attribute.linkParameter)
        }

        return entityAttribute
    }

//region Protobuf to Entities Attribute Parameter mappers

    private fun fromProtobufModel(
        entityAttribute: EntityAttribute,
        protobufParameter: Entitymanager.EntityStringAttributeParameter
    )
        : EntityStringAttributeParameter {
        val parameter = EntityStringAttributeParameter(entityAttribute = entityAttribute)
        if (protobufParameter.hasDefaultValue()) {
            parameter.defaultValue = protobufParameter.defaultValue
        }

        if (protobufParameter.hasMaxLength()) {
            parameter.maxLength = protobufParameter.maxLength
        }

        return parameter
    }

    private fun fromProtobufModel(
        entityAttribute: EntityAttribute,
        protobufParameter: Entitymanager.EntityIntegerAttributeParameter
    )
        : EntityIntegerAttributeParameter {
        val parameter = EntityIntegerAttributeParameter(entityAttribute = entityAttribute)
        if (protobufParameter.hasDefaultValue()) {
            parameter.defaultValue = protobufParameter.defaultValue
        }

        if (protobufParameter.hasMaxValue()) {
            parameter.maxValue = protobufParameter.maxValue
        }

        if (protobufParameter.hasMinValue()) {
            parameter.minValue = protobufParameter.minValue
        }

        return parameter
    }

    private fun fromProtobufModel(
        entityAttribute: EntityAttribute,
        protobufParameter: Entitymanager.EntityDoubleAttributeParameter
    )
        : EntityDoubleAttributeParameter {
        val parameter = EntityDoubleAttributeParameter(entityAttribute = entityAttribute)
        if (protobufParameter.hasDefaultValue()) {
            parameter.defaultValue = protobufParameter.defaultValue
        }

        if (protobufParameter.hasMaxValue()) {
            parameter.maxValue = protobufParameter.maxValue
        }

        if (protobufParameter.hasMinValue()) {
            parameter.minValue = protobufParameter.minValue
        }

        return parameter
    }

    private fun fromProtobufModel(
        entityAttribute: EntityAttribute,
        protobufParameter: Entitymanager.EntityDateAttributeParameter
    )
        : EntityDateAttributeParameter {
        val parameter = EntityDateAttributeParameter(
            entityAttribute = entityAttribute,
            todayIsDefault = protobufParameter.todayIsDefault
        )
        if (protobufParameter.hasDefaultValue()) {
            parameter.defaultValue = protobufParameter.defaultValue.toLocalDate()
        }

        return parameter
    }

    private fun fromProtobufModel(
        entityAttribute: EntityAttribute,
        protobufParameter: Entitymanager.EntityTimeAttributeParameter
    )
        : EntityTimeAttributeParameter {
        val parameter = EntityTimeAttributeParameter(
            entityAttribute = entityAttribute,
            nowIsDefault = protobufParameter.nowIsDefault
        )
        if (protobufParameter.hasDefaultValue()) {
            parameter.defaultValue = protobufParameter.defaultValue.toLocalTime()
        }

        return parameter
    }

    private fun fromProtobufModel(
        entityAttribute: EntityAttribute,
        protobufParameter: Entitymanager.EntityTimestampAttributeParameter
    )
        : EntityTimestampAttributeParameter {
        val parameter = EntityTimestampAttributeParameter(
            entityAttribute = entityAttribute,
            nowIsDefault = protobufParameter.nowIsDefault
        )
        if (protobufParameter.hasDefaultValue()) {
            parameter.defaultValue = protobufParameter.defaultValue.toInstant()
        }

        return parameter
    }

    private fun fromProtobufModel(
        entityAttribute: EntityAttribute,
        protobufParameter: Entitymanager.EntityLinkAttributeParameter
    )
        : EntityLinkAttributeParameter {
        val parameter = EntityLinkAttributeParameter(
            entityAttribute = entityAttribute,
            target = UUID.fromString(protobufParameter.target),
            isMultiple = protobufParameter.isMultiple,
            isFirst = protobufParameter.isFirst
        )

        return parameter
    }

//endregion

    companion object {
        const val INIT_VERSION_NUMBER: Int = 1
    }
}
