package ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.impl

import org.hibernate.Session
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.EntityMetaRepository
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityMeta
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityVersionStatusEnum
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateRepository
import java.util.*

class EntityMetaRepositoryImpl(session: Session) :
    HibernateRepository<EntityMeta, UUID>(session, EntityMeta::class.java), EntityMetaRepository {

    override fun findByOrganizationId(organizationId: Long): List<EntityMeta> {
        val params = mutableListOf<Pair<String, *>>(
            "organizationId" to organizationId,
            "status" to EntityVersionStatusEnum.ACTIVE.value
        )
        val query = getHierarchyQuery()

        return findAll(query, params = params.toTypedArray())
    }

    override fun findByIds(organizationId: Long, metaIds: Collection<UUID>): List<EntityMeta> {
        val params = mutableListOf<Pair<String, *>>(
            "organizationId" to organizationId,
            "status" to EntityVersionStatusEnum.ACTIVE.value
        )
        var query = getHierarchyQuery()
        if (metaIds.isNotEmpty()) {
            query = buildString {
                append(query)
                append(" and entityMeta.id in :ids")
            }
            params.add("ids" to metaIds)
        }

        return findAll(query, params = params.toTypedArray())
    }

    override fun findById(organizationId: Long, entityMetaId: UUID, loadHierarchically: Boolean): EntityMeta? {
        val params = mutableListOf<Pair<String, *>>(
            "organizationId" to organizationId,
            "id" to entityMetaId
        )

        if (loadHierarchically) {
            params.add("status" to EntityVersionStatusEnum.ACTIVE.value)
        }

        val query = buildString {
            append(if (loadHierarchically) getHierarchyQuery() else getSimpleQuery())
            append(" and entityMeta.id = :id")
        }

        return findOne(query, params = params.toTypedArray())
    }

    override fun findByShortNameAndVersion(
        organizationId: Long,
        shortName: String,
        versionNumber: Int?,
        loadHierarchically: Boolean
    ): EntityMeta? {
        val params = mutableListOf<Pair<String, *>>(
            "organizationId" to organizationId,
            "shortName" to shortName
        )

        if (loadHierarchically) {
            params.add("status" to EntityVersionStatusEnum.ACTIVE.value)
        }

        val query = buildString {
            append(if (loadHierarchically) getHierarchyQuery() else getSimpleQuery())
            append(" and entityMeta.shortName = :shortName")
            if (versionNumber != null) {
                append(" and entityMetaVersion.versionNumber = :versionNumber")
                params.add("versionNumber" to versionNumber)
            }
        }

        return findOne(query, params = params.toTypedArray())
    }

    override fun findDraftById(organizationId: Long, entityMetaId: UUID): EntityMeta? {
        val params = mutableListOf<Pair<String, *>>(
            "organizationId" to organizationId,
            "id" to entityMetaId,
            "status" to EntityVersionStatusEnum.DRAFT.value
        )

        val query = buildString {
            append(getHierarchyQuery())
            append(" and entityMeta.id = :id")
        }

        return findOne(query, params = params.toTypedArray())
    }

    private fun getHierarchyQuery() = """
                select distinct entityMeta from EntityMeta as entityMeta
                left join fetch entityMeta.names
                left join fetch entityMeta.entityMetaVersions as entityMetaVersion
                left join fetch entityMetaVersion.attributes as versionAttribute
                left join fetch entityMetaVersion.status
                left join fetch versionAttribute.id.entityAttribute as entityAttribute
                left join fetch entityAttribute.names
                left join fetch entityAttribute.complexType
                left join fetch entityAttribute.integerParameter
                left join fetch entityAttribute.doubleParameter
                left join fetch entityAttribute.stringParameter
                left join fetch entityAttribute.dateParameter
                left join fetch entityAttribute.timeParameter
                left join fetch entityAttribute.timestampParameter
                left join fetch entityAttribute.linkParameter
                where entityMeta.organizationId = :organizationId and entityMeta.isDeleted = false and entityMetaVersion.status.id = :status
            """.trimIndent()

    private fun getSimpleQuery() = """
                from EntityMeta as entityMeta
                where entityMeta.organizationId = :organizationId and entityMeta.isDeleted = false
            """.trimIndent()
}
