package ru.yandex.crm.apphost.kotlin.handlers.entitymanager.ut.entitymeta

import org.junit.jupiter.api.Test
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.impl.LinkedEntitiesManagerImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.model.HierarchyConnection
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityMeta
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityMetaVersion
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.ut.BaseEntityMetaTest
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.ut.entitymeta.mock.HierarchyConnectionManagerMock
import ru.yandex.crm.proto.gallifrey.entitystorage.Entitystorage
import java.util.UUID
import kotlin.test.assertEquals
import kotlin.test.assertTrue

class LinkedEntitiesManagerTests : BaseEntityMetaTest() {

    @Test
    fun `Should return empty list when there is no hierarchy connection`() {
        val linkedEntitiesManager = LinkedEntitiesManagerImpl(HierarchyConnectionManagerMock())

        val connections = linkedEntitiesManager.generateFlatConnections(
            emptyList(),
            getEntities(),
            1
        )

        assertTrue(connections.isEmpty())
    }

    @Test
    fun `Should return empty connections when there is no entities`() {
        val linkedEntitiesManager = LinkedEntitiesManagerImpl(HierarchyConnectionManagerMock())

        val hierarchyConnections = listOf(
            HierarchyConnection(
                1,
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f11"),
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f11"),
                1,
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f12"),
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f12"),
                2,
                isMultiple = true,
                modifiedEntityIsFirst = true
            )
        )

        val connections = linkedEntitiesManager.generateFlatConnections(
            hierarchyConnections,
            getEntities(),
            1
        )

        assertEquals(0, connections.size)
    }

    @Test
    fun `Should return 1 connections when there are not all entities`() {
        val linkedEntitiesManager = LinkedEntitiesManagerImpl(HierarchyConnectionManagerMock())

        val hierarchyConnections = listOf(
            HierarchyConnection(
                1,
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f01"),
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f01"),
                1,
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f12"),
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f12"),
                2,
                isMultiple = true,
                modifiedEntityIsFirst = true
            ),
            HierarchyConnection(
                1,
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f01"),
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f01"),
                1,
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f03"),
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f03"),
                3,
                isMultiple = true,
                modifiedEntityIsFirst = true
            )
        )

        val connections = linkedEntitiesManager.generateFlatConnections(
            hierarchyConnections,
            getEntities(),
            1
        )

        assertEquals(1, connections.size)
    }

    @Test
    fun `Should return 2 connections when parent has 2 childs`() {
        val linkedEntitiesManager = LinkedEntitiesManagerImpl(HierarchyConnectionManagerMock())

        val hierarchyConnections = listOf(
            HierarchyConnection(
                1,
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f01"),
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f01"),
                1,
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f02"),
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f02"),
                2,
                isMultiple = true,
                modifiedEntityIsFirst = true
            ),
            HierarchyConnection(
                1,
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f01"),
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f01"),
                1,
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f03"),
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f03"),
                3,
                isMultiple = true,
                modifiedEntityIsFirst = true
            )
        )

        val connections = linkedEntitiesManager.generateFlatConnections(
            hierarchyConnections,
            getEntities(),
            1
        )

        assertEquals(2, connections.size)
    }

    @Test
    fun `Should return 3 connections when there are A-B and B-C links`() {
        val linkedEntitiesManager = LinkedEntitiesManagerImpl(HierarchyConnectionManagerMock())

        val hierarchyConnections = listOf(
            HierarchyConnection(
                1,
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f01"),
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f01"),
                1,
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f02"),
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f02"),
                2,
                isMultiple = true,
                modifiedEntityIsFirst = true
            ),
            HierarchyConnection(
                1,
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f02"),
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f02"),
                2,
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f03"),
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f03"),
                3,
                isMultiple = true,
                modifiedEntityIsFirst = true
            )
        )

        val connections = linkedEntitiesManager.generateFlatConnections(
            hierarchyConnections,
            getEntities(),
            1
        )

        assertEquals(3, connections.size)
    }

    @Test
    fun `Should return 2 connections when there are child link in DB`() {
        val linkedEntitiesManager = LinkedEntitiesManagerImpl(HierarchyConnectionManagerMock())

        val hierarchyConnections = listOf(
            HierarchyConnection(
                1,
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f01"),
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f01"),
                1,
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f05"),
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f05"),
                5,
                isMultiple = true,
                modifiedEntityIsFirst = true
            )
        )

        val connections = linkedEntitiesManager.generateFlatConnections(
            hierarchyConnections,
            getEntities(),
            1
        )

        assertEquals(2, connections.size)
    }

    @Test
    fun `Should return 2 connections when there are parent link in DB`() {
        val linkedEntitiesManager = LinkedEntitiesManagerImpl(HierarchyConnectionManagerMock())

        val hierarchyConnections = listOf(
            HierarchyConnection(
                1,
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f04"),
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f04"),
                4,
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f01"),
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f01"),
                1,
                isMultiple = true,
                modifiedEntityIsFirst = true
            )
        )

        val connections = linkedEntitiesManager.generateFlatConnections(
            hierarchyConnections,
            getEntities(),
            1
        )

        assertEquals(2, connections.size)
    }

    @Test
    fun `Should return 4 connections when there are parent and child links in DB`() {
        val linkedEntitiesManager = LinkedEntitiesManagerImpl(HierarchyConnectionManagerMock())

        val hierarchyConnections = listOf(
            HierarchyConnection(
                1,
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f04"),
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f04"),
                4,
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f05"),
                UUID.fromString("1c2b66d9-d773-4952-8a06-eb3d69c95f05"),
                5,
                isMultiple = true,
                modifiedEntityIsFirst = true
            )
        )

        val connections = linkedEntitiesManager.generateFlatConnections(
            hierarchyConnections,
            getEntities(),
            1
        )

        assertEquals(4, connections.size)
    }

    @Test
    fun `Should return empty filter when there is no link attributes`() {
        val linkedEntitiesManager = LinkedEntitiesManagerImpl(HierarchyConnectionManagerMock())

        val entityMeta = EntityMeta(
            entityMetaVersions = mutableSetOf(
                EntityMetaVersion(
                    attributes = mutableSetOf(
                        buildStringAttribute("name"),
                        buildIntAttribute("age"),
                        buildStringAttribute("test"),
                    )
                )
            )
        )

        val internalRecordId = 1L

        val filters = linkedEntitiesManager.getLinkFilters(entityMeta, internalRecordId)

        assertTrue(filters.isEmpty())
    }

    @Test
    fun `Should return only filters by first`() {
        val linkedEntitiesManager = LinkedEntitiesManagerImpl(HierarchyConnectionManagerMock())

        val metaId = UUID.randomUUID()
        val entityMeta = EntityMeta(
            id = metaId,
            entityMetaVersions = mutableSetOf(
                EntityMetaVersion(
                    attributes = mutableSetOf(
                        buildStringAttribute("name"),
                        buildIntAttribute("age"),
                        buildStringAttribute("test"),
                        buildLinkAttribute("link1", true),
                        buildLinkAttribute("link2", true),
                    )
                )
            )
        )

        val internalRecordId = 1L

        val filters = linkedEntitiesManager.getLinkFilters(entityMeta, internalRecordId)

        assertEquals(2, filters.size)
        assertTrue(filters.all { it.first != null && it.second == null })
        assertTrue(filters.all { it.first?.metaId == metaId.toString() && it.first?.recordInternalId == internalRecordId })
    }

    @Test
    fun `Should return only filters by second`() {
        val linkedEntitiesManager = LinkedEntitiesManagerImpl(HierarchyConnectionManagerMock())

        val metaId = UUID.randomUUID()
        val entityMeta = EntityMeta(
            id = metaId,
            entityMetaVersions = mutableSetOf(
                EntityMetaVersion(
                    attributes = mutableSetOf(
                        buildStringAttribute("name"),
                        buildIntAttribute("age"),
                        buildStringAttribute("test"),
                        buildLinkAttribute("link1", false),
                        buildLinkAttribute("link2", false),
                    )
                )
            )
        )

        val internalRecordId = 1L

        val filters = linkedEntitiesManager.getLinkFilters(entityMeta, internalRecordId)

        assertEquals(2, filters.size)
        assertTrue(filters.all { it.first == null && it.second != null })
        assertTrue(filters.all { it.second?.metaId == metaId.toString() && it.second?.recordInternalId == internalRecordId })
    }

    @Test
    fun `Should return both types of filters`() {
        val linkedEntitiesManager = LinkedEntitiesManagerImpl(HierarchyConnectionManagerMock())

        val metaId = UUID.randomUUID()
        val entityMeta = EntityMeta(
            id = metaId,
            entityMetaVersions = mutableSetOf(
                EntityMetaVersion(
                    attributes = mutableSetOf(
                        buildStringAttribute("name"),
                        buildIntAttribute("age"),
                        buildStringAttribute("test"),
                        buildLinkAttribute("link1", true),
                        buildLinkAttribute("link2", false),
                    )
                )
            )
        )

        val internalRecordId = 1L

        val filters = linkedEntitiesManager.getLinkFilters(entityMeta, internalRecordId)

        assertEquals(2, filters.size)
        assertTrue(filters.any { it.first != null })
        assertTrue(filters.any { it.second != null })
    }

    private fun getEntities() = listOf(
        Entitystorage.EntityRecord.newBuilder()
            .setId("1c2b66d9-d773-4952-8a06-eb3d69c95f01")
            .setMetaId("1c2b66d9-d773-4952-8a06-eb3d69c95f01")
            .setInternalId(1)
            .build(),

        Entitystorage.EntityRecord.newBuilder()
            .setId("1c2b66d9-d773-4952-8a06-eb3d69c95f02")
            .setMetaId("1c2b66d9-d773-4952-8a06-eb3d69c95f02")
            .setInternalId(2)
            .build(),

        Entitystorage.EntityRecord.newBuilder()
            .setId("1c2b66d9-d773-4952-8a06-eb3d69c95f03")
            .setMetaId("1c2b66d9-d773-4952-8a06-eb3d69c95f03")
            .setInternalId(3)
            .build(),

        Entitystorage.EntityRecord.newBuilder()
            .setId("1c2b66d9-d773-4952-8a06-eb3d69c95f04")
            .setMetaId("1c2b66d9-d773-4952-8a06-eb3d69c95f04")
            .setInternalId(4)
            .build(),

        Entitystorage.EntityRecord.newBuilder()
            .setId("1c2b66d9-d773-4952-8a06-eb3d69c95f05")
            .setMetaId("1c2b66d9-d773-4952-8a06-eb3d69c95f05")
            .setInternalId(5)
            .build(),

        Entitystorage.EntityRecord.newBuilder()
            .setId("1c2b66d9-d773-4952-8a06-eb3d69c95f06")
            .setMetaId("1c2b66d9-d773-4952-8a06-eb3d69c95f06")
            .setInternalId(6)
            .build(),

        Entitystorage.EntityRecord.newBuilder()
            .setId("1c2b66d9-d773-4952-8a06-eb3d69c95f07")
            .setMetaId("1c2b66d9-d773-4952-8a06-eb3d69c95f07")
            .setInternalId(5)
            .build(),
    )
}
