package ru.yandex.crm.apphost.kotlin.handlers.entitymanager.ut.entitymeta.mappers

import org.junit.jupiter.api.Test
import org.koin.test.KoinTest
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.mappers.impl.AttributeValueProviderImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.model.SimpleTypeEnum
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.ComplexType
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityAttribute
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.SimpleType
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.parameter.*
import java.time.Instant
import java.time.LocalDate
import java.time.LocalTime
import java.util.*
import kotlin.test.assertEquals
import kotlin.test.assertNotNull
import kotlin.test.assertNull

class AttributeValueProviderTests : KoinTest {

    @Test
    fun `Should return int value`() {
        val valueProvider = AttributeValueProviderImpl()
        val defaultValue = 100

        val value = "123"
        val attribute = EntityAttribute(
            integerParameter = EntityIntegerAttributeParameter(defaultValue = defaultValue),
            complexType = ComplexType(name = "INT", simpleType = SimpleType(SimpleTypeEnum.INT.value))
        )

        val convertedValue = valueProvider.getValue(attribute, value)
        assertNotNull(convertedValue)
        assertEquals(123, convertedValue)
    }

    @Test
    fun `Should return default int value`() {
        val valueProvider = AttributeValueProviderImpl()
        val defaultValue = 100

        val value = ""
        val attribute = EntityAttribute(
            integerParameter = EntityIntegerAttributeParameter(defaultValue = defaultValue),
            complexType = ComplexType(name = "INT", simpleType = SimpleType(SimpleTypeEnum.INT.value))
        )

        val convertedValue = valueProvider.getValue(attribute, value)
        assertNotNull(convertedValue)
        assertEquals(defaultValue, convertedValue)
    }

    @Test
    fun `Should return null for int`() {
        val valueProvider = AttributeValueProviderImpl()

        val value = ""
        val attribute = EntityAttribute(
            integerParameter = EntityIntegerAttributeParameter(),
            complexType = ComplexType(name = "INT", simpleType = SimpleType(SimpleTypeEnum.INT.value))
        )

        val convertedValue = valueProvider.getValue(attribute, value)
        assertNull(convertedValue)
    }

    @Test
    fun `Should return double value`() {
        val valueProvider = AttributeValueProviderImpl()
        val defaultValue = 100.0

        val value = "123"
        val attribute = EntityAttribute(
            doubleParameter = EntityDoubleAttributeParameter(defaultValue = defaultValue),
            complexType = ComplexType(name = "DOUBLE", simpleType = SimpleType(SimpleTypeEnum.DOUBLE.value))
        )

        val convertedValue = valueProvider.getValue(attribute, value)
        assertNotNull(convertedValue)
        assertEquals(123.0, convertedValue)
    }

    @Test
    fun `Should return default double value`() {
        val valueProvider = AttributeValueProviderImpl()
        val defaultValue = 100.0

        val value = ""
        val attribute = EntityAttribute(
            doubleParameter = EntityDoubleAttributeParameter(defaultValue = defaultValue),
            complexType = ComplexType(name = "DOUBLE", simpleType = SimpleType(SimpleTypeEnum.DOUBLE.value))
        )

        val convertedValue = valueProvider.getValue(attribute, value)
        assertNotNull(convertedValue)
        assertEquals(defaultValue, convertedValue)
    }

    @Test
    fun `Should return null for double`() {
        val valueProvider = AttributeValueProviderImpl()

        val value = ""
        val attribute = EntityAttribute(
            doubleParameter = EntityDoubleAttributeParameter(),
            complexType = ComplexType(name = "DOUBLE", simpleType = SimpleType(SimpleTypeEnum.DOUBLE.value))
        )

        val convertedValue = valueProvider.getValue(attribute, value)
        assertNull(convertedValue)
    }

    @Test
    fun `Should return string value`() {
        val valueProvider = AttributeValueProviderImpl()
        val defaultValue = "test"

        val value = "100"
        val attribute = EntityAttribute(
            stringParameter = EntityStringAttributeParameter(defaultValue = defaultValue),
            complexType = ComplexType(name = "STRING", simpleType = SimpleType(SimpleTypeEnum.STRING.value))
        )

        val convertedValue = valueProvider.getValue(attribute, value)
        assertNotNull(convertedValue)
        assertEquals("100", convertedValue)
    }

    @Test
    fun `Should return default string value`() {
        val valueProvider = AttributeValueProviderImpl()
        val defaultValue = "test"

        val value = ""
        val attribute = EntityAttribute(
            stringParameter = EntityStringAttributeParameter(defaultValue = defaultValue),
            complexType = ComplexType(name = "STRING", simpleType = SimpleType(SimpleTypeEnum.STRING.value))
        )

        val convertedValue = valueProvider.getValue(attribute, value)
        assertNotNull(convertedValue)
        assertEquals(defaultValue, convertedValue)
    }

    @Test
    fun `Should return null for string`() {
        val valueProvider = AttributeValueProviderImpl()

        val value = ""
        val attribute = EntityAttribute(
            stringParameter = EntityStringAttributeParameter(),
            complexType = ComplexType(name = "STRING", simpleType = SimpleType(SimpleTypeEnum.STRING.value))
        )

        val convertedValue = valueProvider.getValue(attribute, value)
        assertNull(convertedValue)
    }

    @Test
    fun `Should return string date`() {
        val valueProvider = AttributeValueProviderImpl()
        val defaultValue = LocalDate.parse("2022-01-02")

        val value = "2000-01-02"
        val attribute = EntityAttribute(
            dateParameter = EntityDateAttributeParameter(defaultValue = defaultValue),
            complexType = ComplexType(name = "DATE", simpleType = SimpleType(SimpleTypeEnum.DATE.value))
        )

        val convertedValue = valueProvider.getValue(attribute, value)
        assertNotNull(convertedValue)
        assertEquals(LocalDate.parse("2000-01-02"), convertedValue)
    }

    @Test
    fun `Should return default date value`() {
        val valueProvider = AttributeValueProviderImpl()
        val defaultValue = LocalDate.parse("2022-01-02")

        val value = ""
        val attribute = EntityAttribute(
            dateParameter = EntityDateAttributeParameter(defaultValue = defaultValue),
            complexType = ComplexType(name = "DATE", simpleType = SimpleType(SimpleTypeEnum.DATE.value))
        )

        val convertedValue = valueProvider.getValue(attribute, value)
        assertNotNull(convertedValue)
        assertEquals(defaultValue, convertedValue)
    }

    @Test
    fun `Should return null for date`() {
        val valueProvider = AttributeValueProviderImpl()

        val value = ""
        val attribute = EntityAttribute(
            dateParameter = EntityDateAttributeParameter(),
            complexType = ComplexType(name = "DATE", simpleType = SimpleType(SimpleTypeEnum.DATE.value))
        )

        val convertedValue = valueProvider.getValue(attribute, value)
        assertNull(convertedValue)
    }

    @Test
    fun `Should return not-null for date`() {
        val valueProvider = AttributeValueProviderImpl()

        val value = ""
        val attribute = EntityAttribute(
            dateParameter = EntityDateAttributeParameter(todayIsDefault = true),
            complexType = ComplexType(name = "DATE", simpleType = SimpleType(SimpleTypeEnum.DATE.value))
        )

        val convertedValue = valueProvider.getValue(attribute, value)
        assertNotNull(convertedValue)
    }

    @Test
    fun `Should return string time`() {
        val valueProvider = AttributeValueProviderImpl()
        val defaultValue = LocalTime.parse("18:00")

        val value = "12:00"
        val attribute = EntityAttribute(
            timeParameter = EntityTimeAttributeParameter(defaultValue = defaultValue),
            complexType = ComplexType(name = "TIME", simpleType = SimpleType(SimpleTypeEnum.TIME.value))
        )

        val convertedValue = valueProvider.getValue(attribute, value)
        assertNotNull(convertedValue)
        assertEquals(LocalTime.parse("12:00"), convertedValue)
    }

    @Test
    fun `Should return default date time`() {
        val valueProvider = AttributeValueProviderImpl()
        val defaultValue = LocalTime.parse("18:00")

        val value = ""
        val attribute = EntityAttribute(
            timeParameter = EntityTimeAttributeParameter(defaultValue = defaultValue),
            complexType = ComplexType(name = "TIME", simpleType = SimpleType(SimpleTypeEnum.TIME.value))
        )

        val convertedValue = valueProvider.getValue(attribute, value)
        assertNotNull(convertedValue)
        assertEquals(defaultValue, convertedValue)
    }

    @Test
    fun `Should return null for time`() {
        val valueProvider = AttributeValueProviderImpl()

        val value = ""
        val attribute = EntityAttribute(
            timeParameter = EntityTimeAttributeParameter(),
            complexType = ComplexType(name = "TIME", simpleType = SimpleType(SimpleTypeEnum.TIME.value))
        )

        val convertedValue = valueProvider.getValue(attribute, value)
        assertNull(convertedValue)
    }

    @Test
    fun `Should return not-null for time`() {
        val valueProvider = AttributeValueProviderImpl()

        val value = ""
        val attribute = EntityAttribute(
            timeParameter = EntityTimeAttributeParameter(nowIsDefault = true),
            complexType = ComplexType(name = "TIME", simpleType = SimpleType(SimpleTypeEnum.TIME.value))
        )

        val convertedValue = valueProvider.getValue(attribute, value)
        assertNotNull(convertedValue)
    }

    @Test
    fun `Should return string timestamp`() {
        val valueProvider = AttributeValueProviderImpl()
        val defaultValue = Instant.parse("1990-12-01T14:00:00Z")

        val value = "2020-12-01T14:00:00Z"
        val attribute = EntityAttribute(
            timestampParameter = EntityTimestampAttributeParameter(defaultValue = defaultValue),
            complexType = ComplexType(name = "TIMESTAMP", simpleType = SimpleType(SimpleTypeEnum.TIMESTAMP.value))
        )

        val convertedValue = valueProvider.getValue(attribute, value)
        assertNotNull(convertedValue)
        assertEquals(Instant.parse("2020-12-01T14:00:00Z"), convertedValue)
    }

    @Test
    fun `Should return default date timestamp`() {
        val valueProvider = AttributeValueProviderImpl()
        val defaultValue = Instant.parse("1990-12-01T14:00:00Z")

        val value = ""
        val attribute = EntityAttribute(
            timestampParameter = EntityTimestampAttributeParameter(defaultValue = defaultValue),
            complexType = ComplexType(name = "TIMESTAMP", simpleType = SimpleType(SimpleTypeEnum.TIMESTAMP.value))
        )

        val convertedValue = valueProvider.getValue(attribute, value)
        assertNotNull(convertedValue)
        assertEquals(defaultValue, convertedValue)
    }

    @Test
    fun `Should return null for timestamp`() {
        val valueProvider = AttributeValueProviderImpl()

        val value = ""
        val attribute = EntityAttribute(
            timestampParameter = EntityTimestampAttributeParameter(),
            complexType = ComplexType(name = "TIMESTAMP", simpleType = SimpleType(SimpleTypeEnum.TIMESTAMP.value))
        )

        val convertedValue = valueProvider.getValue(attribute, value)
        assertNull(convertedValue)
    }

    @Test
    fun `Should return not null for timestamp`() {
        val valueProvider = AttributeValueProviderImpl()

        val value = ""
        val attribute = EntityAttribute(
            timestampParameter = EntityTimestampAttributeParameter(nowIsDefault = true),
            complexType = ComplexType(name = "TIMESTAMP", simpleType = SimpleType(SimpleTypeEnum.TIMESTAMP.value))
        )

        val convertedValue = valueProvider.getValue(attribute, value)
        assertNotNull(convertedValue)
    }

    @Test
    fun `Should return true for bool`() {
        val valueProvider = AttributeValueProviderImpl()

        val value = "true"
        val attribute = EntityAttribute(
            complexType = ComplexType(name = "BOOL", simpleType = SimpleType(SimpleTypeEnum.BOOL.value))
        )

        val convertedValue = valueProvider.getValue(attribute, value)
        assertEquals(true, convertedValue)
    }

    @Test
    fun `Should return false for bool`() {
        val valueProvider = AttributeValueProviderImpl()

        val value = "test"
        val attribute = EntityAttribute(
            complexType = ComplexType(name = "BOOL", simpleType = SimpleType(SimpleTypeEnum.BOOL.value))
        )

        val convertedValue = valueProvider.getValue(attribute, value)
        assertEquals(false, convertedValue)
    }

    @Test
    fun `Should return UUID value`() {
        val valueProvider = AttributeValueProviderImpl()

        val uuidValue = UUID.randomUUID()

        val value = uuidValue.toString()
        val attribute = EntityAttribute(
            complexType = ComplexType(name = "UUID", simpleType = SimpleType(SimpleTypeEnum.UUID.value))
        )

        val convertedValue = valueProvider.getValue(attribute, value)
        assertEquals(uuidValue, convertedValue)
    }

    @Test
    fun `Should return null for UUID`() {
        val valueProvider = AttributeValueProviderImpl()

        val value = ""
        val attribute = EntityAttribute(
            complexType = ComplexType(name = "UUID", simpleType = SimpleType(SimpleTypeEnum.UUID.value))
        )

        val convertedValue = valueProvider.getValue(attribute, value)
        assertNull(convertedValue)
    }
}
