package ru.yandex.crm.apphost.kotlin.handlers.entitymanager.ut.entitymeta.validator

import org.junit.jupiter.api.Test
import org.koin.test.KoinTest
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.impl.DateAttributeValidatorImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.model.ValidationResultReason
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityAttribute
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityMetaVersionAttribute
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityMetaVersionAttributeId
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.parameter.EntityDateAttributeParameter
import java.time.LocalDate
import kotlin.test.assertEquals
import kotlin.test.assertFalse
import kotlin.test.assertTrue

class DateAttributeValidatorTests : KoinTest {

    @Test
    fun `Should return success`() {
        val validator = DateAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = true,
                isNullable = false,
                dateParameter = EntityDateAttributeParameter()
            )
            ),
            "test"
        )

        val value = "2022-01-02"

        val validationResult = validator.validate(value, attribute)

        assertTrue(validationResult.isValid)
    }

    @Test
    fun `Should return success when attribute is not required but value is empty`() {
        val validator = DateAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = false,
                isNullable = true,
                dateParameter = EntityDateAttributeParameter()
            )
            ),
            "test"
        )

        val value = ""
        val validationResult = validator.validate(value, attribute)

        assertTrue(validationResult.isValid)
    }

    @Test
    fun `Should return error when attribute is required but value is empty`() {
        val validator = DateAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = true,
                isNullable = false,
                dateParameter = EntityDateAttributeParameter()
            )
            ),
            "test"
        )

        val value = ""
        val validationResult = validator.validate(value, attribute)

        assertFalse(validationResult.isValid)
        assertEquals(ValidationResultReason.REQUIRED, validationResult.reason)
    }

    @Test
    fun `Should return error when date has incorrect format`() {
        val validator = DateAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = true,
                isNullable = false,
                dateParameter = EntityDateAttributeParameter()
            )
            ),
            "test"
        )

        val value = "test"
        val validationResult = validator.validate(value, attribute)

        assertFalse(validationResult.isValid)
        assertEquals(ValidationResultReason.INCORRECT_FORMAT, validationResult.reason)
    }

    @Test
    fun `Should return success when attribute is required but there is default value`() {
        val validator = DateAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = true,
                isNullable = false,
                dateParameter = EntityDateAttributeParameter(
                    defaultValue = LocalDate.now()
                )
            )),
            "test"
        )

        val value = ""
        val validationResult = validator.validate(value, attribute)

        assertTrue(validationResult.isValid)
    }

    @Test
    fun `Should return success when attribute is required but today is default value`() {
        val validator = DateAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = true,
                isNullable = false,
                dateParameter = EntityDateAttributeParameter(
                   todayIsDefault = true
                )
            )),
            "test"
        )

        val value = ""
        val validationResult = validator.validate(value, attribute)

        assertTrue(validationResult.isValid)
    }
}
