package ru.yandex.crm.apphost.kotlin.handlers.entitymanager.ut.entitymeta.validator

import org.junit.jupiter.api.Test
import org.koin.test.KoinTest
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.impl.DoubleAttributeValidatorImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.model.ValidationResultReason
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityAttribute
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityMetaVersionAttribute
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityMetaVersionAttributeId
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.parameter.EntityDoubleAttributeParameter
import kotlin.test.assertEquals
import kotlin.test.assertFalse
import kotlin.test.assertTrue

class DoubleAttributeValidatorTests : KoinTest {

    @Test
    fun `Should return success`() {
        val validator = DoubleAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = true,
                isNullable = false,
                doubleParameter = EntityDoubleAttributeParameter()
            )
            ),
            "test"
        )

        val value = "123"
        val validationResult = validator.validate(value, attribute)

        assertTrue(validationResult.isValid)
    }

    @Test
    fun `Should return success when attribute is not required but value is empty`() {
        val validator = DoubleAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = false,
                isNullable = true,
                doubleParameter = EntityDoubleAttributeParameter()
            )
            ),
            "test"
        )

        val value = ""
        val validationResult = validator.validate(value, attribute)

        assertTrue(validationResult.isValid)
    }

    @Test
    fun `Should return error when attribute is required but value is empty`() {
        val validator = DoubleAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = true,
                isNullable = false,
                doubleParameter = EntityDoubleAttributeParameter()
            )
            ),
            "test"
        )

        val value = ""
        val validationResult = validator.validate(value, attribute)

        assertFalse(validationResult.isValid)
        assertEquals(ValidationResultReason.REQUIRED, validationResult.reason)
    }

    @Test
    fun `Should return success when attribute is required but there is default value`() {
        val validator = DoubleAttributeValidatorImpl()
        val defaultValue = 123.0

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = true,
                isNullable = false,
                doubleParameter = EntityDoubleAttributeParameter(
                    defaultValue = defaultValue
                )
            )),
            "test"
        )

        val value = ""
        val validationResult = validator.validate(value, attribute)

        assertTrue(validationResult.isValid)
    }

    @Test
    fun `Should return error when value has incorrect format`() {
        val validator = DoubleAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = true,
                isNullable = false,
                doubleParameter = EntityDoubleAttributeParameter()
            )
            ),
            "test"
        )

        val value = "test"
        val validationResult = validator.validate(value, attribute)

        assertFalse(validationResult.isValid)
        assertEquals(ValidationResultReason.INCORRECT_FORMAT, validationResult.reason)
    }

    @Test
    fun `Should return error when value is more than max value`() {
        val validator = DoubleAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = true,
                isNullable = false,
                doubleParameter = EntityDoubleAttributeParameter(
                    maxValue = 100.0
                )
            )),
            "test"
        )

        val validationResult = validator.validate("200", attribute)

        assertFalse(validationResult.isValid)
        assertEquals(ValidationResultReason.MORE_THAN_MAX_VALUE, validationResult.reason)
    }

    @Test
    fun `Should return error when value is less than min value`() {
        val validator = DoubleAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = true,
                isNullable = false,
                doubleParameter = EntityDoubleAttributeParameter(
                    minValue = 100.0
                )
            )),
            "test"
        )

        val validationResult = validator.validate("10", attribute)

        assertFalse(validationResult.isValid)
        assertEquals(ValidationResultReason.LESS_THAN_MIN_VALUE, validationResult.reason)
    }
}
