package ru.yandex.crm.apphost.kotlin.handlers.entitymanager.ut.entitymeta.validator

import org.junit.jupiter.api.AfterEach
import org.junit.jupiter.api.BeforeEach
import org.junit.jupiter.api.Test
import org.koin.core.context.startKoin
import org.koin.core.context.stopKoin
import org.koin.dsl.bind
import org.koin.dsl.binds
import org.koin.dsl.module
import org.koin.test.inject
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.BooleanAttributeValidator
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.ComplexAttributeValidator
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.DateAttributeValidator
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.DoubleAttributeValidator
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.EmailAttributeValidator
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.EntityValidator
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.IntAttributeValidator
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.PhoneAttributeValidator
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.SimpleAttributeValidator
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.StringAttributeValidator
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.TimeAttributeValidator
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.TimestampAttributeValidator
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.UuidAttributeValidator
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.ValidatorFactory
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.impl.BooleanAttributeValidatorImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.impl.DateAttributeValidatorImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.impl.DoubleAttributeValidatorImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.impl.EmailAttributeValidatorImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.impl.EntityValidatorImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.impl.IntAttributeValidatorImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.impl.PhoneAttributeValidatorImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.impl.StringAttributeValidatorImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.impl.TimeAttributeValidatorImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.impl.TimestampAttributeValidatorImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.impl.UuidAttributeValidatorImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.impl.ValidatorFactoryImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityMeta
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityMetaVersion
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.ut.BaseEntityMetaTest
import ru.yandex.crm.proto.gallifrey.entitymanager.Entitymanager
import kotlin.test.assertFalse
import kotlin.test.assertTrue

private val validationModule = module {
    single { BooleanAttributeValidatorImpl() } binds arrayOf(BooleanAttributeValidator::class, SimpleAttributeValidator::class)
    single { DateAttributeValidatorImpl() } binds arrayOf(DateAttributeValidator::class, SimpleAttributeValidator::class)
    single { DoubleAttributeValidatorImpl() } binds arrayOf(DoubleAttributeValidator::class, SimpleAttributeValidator::class)
    single { IntAttributeValidatorImpl() } binds arrayOf(IntAttributeValidator::class, SimpleAttributeValidator::class)
    single { StringAttributeValidatorImpl() } binds arrayOf(StringAttributeValidator::class, SimpleAttributeValidator::class)
    single { TimeAttributeValidatorImpl() } binds arrayOf(TimeAttributeValidator::class, SimpleAttributeValidator::class)
    single { TimestampAttributeValidatorImpl() } binds arrayOf(TimestampAttributeValidator::class, SimpleAttributeValidator::class)
    single { UuidAttributeValidatorImpl() } binds arrayOf(UuidAttributeValidator::class, SimpleAttributeValidator::class)
    single { EmailAttributeValidatorImpl() } binds arrayOf(EmailAttributeValidator::class, ComplexAttributeValidator::class)
    single { PhoneAttributeValidatorImpl() } binds arrayOf(PhoneAttributeValidator::class, ComplexAttributeValidator::class)
    single { ValidatorFactoryImpl() } bind ValidatorFactory::class
    single { EntityValidatorImpl(get()) } bind EntityValidator::class
}

class EntityValidatorTests : BaseEntityMetaTest() {
    @BeforeEach
    fun beforeEach() {
        startKoin {
            modules(validationModule)
        }
    }

    @AfterEach
    fun afterEach() {
        stopKoin()
    }

    @Test
    fun `Should return success`() {
        val validator by inject<EntityValidator>()

        val properties = mutableListOf(
            buildAttribute("name", "Name"),
            buildAttribute("age", "18"),
            buildAttribute("test", "test")
        )

        val entityMeta = EntityMeta(
            entityMetaVersions = mutableSetOf(
                EntityMetaVersion(
                    attributes = mutableSetOf(
                        buildStringAttribute("name"),
                        buildIntAttribute("age"),
                        buildStringAttribute("test")
                    )
                )
            )
        )

        val validationResult = validator.validate(entityMeta, properties)

        assertTrue(validationResult.isSuccess)
    }

    @Test
    fun `Should return error when there are more attributes`() {
        val validator by inject<EntityValidator>()

        val properties = mutableListOf(
            buildAttribute("name", "Name"),
            buildAttribute("age", "18"),
            buildAttribute("test", "test")
        )

        val entityMeta = EntityMeta(
            entityMetaVersions = mutableSetOf(
                EntityMetaVersion(
                    attributes = mutableSetOf(
                        buildStringAttribute("name"),
                        buildIntAttribute("age")
                    )
                )
            )
        )

        val validationResult = validator.validate(entityMeta, properties)

        assertFalse(validationResult.isSuccess)
    }

    @Test
    fun `Should return error when required attributes are skipped`() {
        val validator by inject<EntityValidator>()

        val properties = mutableListOf(
            buildAttribute("name", "Name"),
            buildAttribute("age", "18")
        )

        val entityMeta = EntityMeta(
            entityMetaVersions = mutableSetOf(
                EntityMetaVersion(
                    attributes = mutableSetOf(
                        buildStringAttribute("name"),
                        buildIntAttribute("age"),
                        buildStringAttribute("test")
                    )
                )
            )
        )

        val validationResult = validator.validate(entityMeta, properties)

        assertFalse(validationResult.isSuccess)
    }

    @Test
    fun `Should return success when not required attributes are skipped`() {
        val validator by inject<EntityValidator>()

        val properties = mutableListOf(
            buildAttribute("name", "Name"),
            buildAttribute("age", "18")
        )

        val entityMeta = EntityMeta(
            entityMetaVersions = mutableSetOf(
                EntityMetaVersion(
                    attributes = mutableSetOf(
                        buildStringAttribute("name"),
                        buildIntAttribute("age"),
                        buildStringAttribute("test", false, true)
                    )
                )
            )
        )

        val validationResult = validator.validate(entityMeta, properties)

        assertTrue(validationResult.isSuccess)
    }

    @Test
    fun `Should return error when attributes have validation error`() {
        val validator by inject<EntityValidator>()

        val properties = mutableListOf(
            buildAttribute("name", "Name"),
            buildAttribute("age", "18"),
            buildAttribute("test", "test")
        )

        val entityMeta = EntityMeta(
            entityMetaVersions = mutableSetOf(
                EntityMetaVersion(
                    attributes = mutableSetOf(
                        buildStringAttribute("name"),
                        buildIntAttribute("age"),
                        buildEmailAttribute("test")
                    )
                )
            )
        )

        val validationResult = validator.validate(entityMeta, properties)

        assertFalse(validationResult.isSuccess)
    }

    private fun buildAttribute(name: String, value: String)
        : Entitymanager.EntityRecordAttribute {
        return Entitymanager.EntityRecordAttribute.newBuilder()
            .setName(name)
            .setValue(value)
            .build()
    }
}
