package ru.yandex.crm.apphost.kotlin.handlers.entitymanager.ut.entitymeta.validator

import org.junit.jupiter.api.AfterEach
import org.junit.jupiter.api.BeforeEach
import org.junit.jupiter.api.Test
import org.koin.core.context.startKoin
import org.koin.core.context.stopKoin
import org.koin.dsl.bind
import org.koin.dsl.module
import org.koin.test.KoinTest
import org.koin.test.inject
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.EmailAttributeValidator
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.PhoneAttributeValidator
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.StringAttributeValidator
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.impl.EmailAttributeValidatorImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.impl.PhoneAttributeValidatorImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.impl.StringAttributeValidatorImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.model.ValidationResultReason
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityAttribute
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityMetaVersionAttribute
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityMetaVersionAttributeId
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.parameter.EntityStringAttributeParameter
import kotlin.test.assertEquals
import kotlin.test.assertFalse
import kotlin.test.assertTrue

private val validationModule = module {
    single { StringAttributeValidatorImpl() } bind StringAttributeValidator::class
    single { PhoneAttributeValidatorImpl() } bind PhoneAttributeValidator::class
}

class PhoneAttributeValidatorTests : KoinTest {
    @BeforeEach
    fun beforeEach() {
        startKoin {
            modules(validationModule)
        }
    }

    @AfterEach
    fun afterEach() {
        stopKoin()
    }

    @Test
    fun `Should return success`() {
        val validator by inject<PhoneAttributeValidator>()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(
                entityAttribute = EntityAttribute(
                    isRequired = true,
                    isNullable = false,
                    stringParameter = EntityStringAttributeParameter(
                        maxLength = 200
                    )
                )
            ),
            "test"
        )

        val value = "+77777777777"
        val validationResult = validator.validate(value, attribute)

        assertTrue(validationResult.isValid)
    }

    @Test
    fun `Should return success when attribute is not required but value is empty`() {
        val validator by inject<PhoneAttributeValidator>()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(
                entityAttribute = EntityAttribute(
                    isRequired = false,
                    isNullable = true,
                    stringParameter = EntityStringAttributeParameter()
                )
            ),
            "test"
        )

        val value = ""
        val validationResult = validator.validate(value, attribute)

        assertTrue(validationResult.isValid)
    }

    @Test
    fun `Should return error when attribute is required but value is empty`() {
        val validator by inject<PhoneAttributeValidator>()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(
                entityAttribute = EntityAttribute(
                    isRequired = true,
                    isNullable = false,
                    stringParameter = EntityStringAttributeParameter()
                )
            ),
            "test"
        )

        val value = ""
        val validationResult = validator.validate(value, attribute)

        assertFalse(validationResult.isValid)
        assertEquals(ValidationResultReason.REQUIRED, validationResult.reason)
    }

    @Test
    fun `Should return error when value has incorrect format`() {
        val validator by inject<PhoneAttributeValidator>()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(
                entityAttribute = EntityAttribute(
                    isRequired = true,
                    isNullable = false,
                    stringParameter = EntityStringAttributeParameter()
                )
            ),
            "test"
        )

        val value = "test"
        val validationResult = validator.validate(value, attribute)

        assertFalse(validationResult.isValid)
        assertEquals(ValidationResultReason.INCORRECT_FORMAT, validationResult.reason)
    }
}
