package ru.yandex.crm.apphost.kotlin.handlers.entitymanager.ut.entitymeta.validator

import org.junit.jupiter.api.Test
import org.koin.test.KoinTest
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.impl.StringAttributeValidatorImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.model.ValidationResultReason
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityAttribute
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityMetaVersionAttribute
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityMetaVersionAttributeId
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.parameter.EntityStringAttributeParameter
import kotlin.test.assertEquals
import kotlin.test.assertFalse
import kotlin.test.assertTrue

class StringAttributeValidatorTests : KoinTest {

    @Test
    fun `Should return success`() {
        val validator = StringAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = true,
                isNullable = false,
                stringParameter = EntityStringAttributeParameter(
                    maxLength = 50
                )
            )),
            "test"
        )

        val value = "test"
        val validationResult = validator.validate(value, attribute)

        assertTrue(validationResult.isValid)
    }

    @Test
    fun `Should return success when attribute is not required but value is empty`() {
        val validator = StringAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = false,
                isNullable = true,
                stringParameter = EntityStringAttributeParameter()
            )),
            "test"
        )

        val value = ""
        val validationResult = validator.validate(value, attribute)

        assertTrue(validationResult.isValid)
    }

    @Test
    fun `Should return error when attribute is required but value is empty`() {
        val validator = StringAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = true,
                isNullable = false,
                stringParameter = EntityStringAttributeParameter()
            )),
            "test"
        )

        val value = ""
        val validationResult = validator.validate(value, attribute)

        assertFalse(validationResult.isValid)
        assertEquals(ValidationResultReason.REQUIRED, validationResult.reason)
    }

    @Test
    fun `Should return success when attribute is required but there is default value`() {
        val validator = StringAttributeValidatorImpl()
        val defaultValue = "defaultValue"

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = true,
                isNullable = false,
                stringParameter = EntityStringAttributeParameter(
                    defaultValue = defaultValue
                )
            )),
            "test"
        )

        val value = ""
        val validationResult = validator.validate(value, attribute)

        assertTrue(validationResult.isValid)
    }

    @Test
    fun `Should return error when length is more than max length`() {
        val validator = StringAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = true,
                isNullable = false,
                stringParameter = EntityStringAttributeParameter(
                    maxLength = 2
                )
            )),
            "test"
        )

        val validationResult = validator.validate("test", attribute)

        assertFalse(validationResult.isValid)
        assertEquals(ValidationResultReason.LENGTH_MORE_THAN_MAX_LENGTH, validationResult.reason)
    }
}
