package ru.yandex.crm.apphost.kotlin.handlers.entitymanager.ut.entitymeta.validator

import org.junit.jupiter.api.Test
import org.koin.test.KoinTest
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.impl.TimeAttributeValidatorImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.model.ValidationResultReason
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityAttribute
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityMetaVersionAttribute
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityMetaVersionAttributeId
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.parameter.EntityTimeAttributeParameter
import java.time.LocalTime
import kotlin.test.assertEquals
import kotlin.test.assertFalse
import kotlin.test.assertTrue

class TimeAttributeValidatorTests : KoinTest {

    @Test
    fun `Should return success`() {
        val validator = TimeAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = true,
                isNullable = false,
                timeParameter = EntityTimeAttributeParameter()
            )
            ),
            "test"
        )

        val value = "18:00:00"

        val validationResult = validator.validate(value, attribute)

        assertTrue(validationResult.isValid)
    }

    @Test
    fun `Should return success when attribute is not required but value is empty`() {
        val validator = TimeAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = false,
                isNullable = true,
                timeParameter = EntityTimeAttributeParameter()
            )
            ),
            "test"
        )

        val value = ""
        val validationResult = validator.validate(value, attribute)

        assertTrue(validationResult.isValid)
    }

    @Test
    fun `Should return error when attribute is required but value is empty`() {
        val validator = TimeAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = true,
                isNullable = false,
                timeParameter = EntityTimeAttributeParameter()
            )
            ),
            "test"
        )

        val value = ""
        val validationResult = validator.validate(value, attribute)

        assertFalse(validationResult.isValid)
        assertEquals(ValidationResultReason.REQUIRED, validationResult.reason)
    }

    @Test
    fun `Should return error when date has incorrect format`() {
        val validator = TimeAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = true,
                isNullable = false,
                timeParameter = EntityTimeAttributeParameter()
            )
            ),
            "test"
        )

        val value = "test"
        val validationResult = validator.validate(value, attribute)

        assertFalse(validationResult.isValid)
        assertEquals(ValidationResultReason.INCORRECT_FORMAT, validationResult.reason)
    }

    @Test
    fun `Should return success when attribute is required but there is default value`() {
        val validator = TimeAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = true,
                isNullable = false,
                timeParameter = EntityTimeAttributeParameter(
                    defaultValue = LocalTime.now()
                )
            )),
            "test"
        )

        val value = ""
        val validationResult = validator.validate(value, attribute)

        assertTrue(validationResult.isValid)
    }

    @Test
    fun `Should return success when attribute is required but now is default value`() {
        val validator = TimeAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = true,
                isNullable = false,
                timeParameter = EntityTimeAttributeParameter(
                   nowIsDefault = true
                )
            )),
            "test"
        )

        val value = ""
        val validationResult = validator.validate(value, attribute)

        assertTrue(validationResult.isValid)
    }
}
