package ru.yandex.crm.apphost.kotlin.handlers.entitymanager.ut.entitymeta.validator

import org.junit.jupiter.api.Test
import org.koin.test.KoinTest
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.impl.TimestampAttributeValidatorImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.entitymeta.validator.model.ValidationResultReason
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityAttribute
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityMetaVersionAttribute
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.EntityMetaVersionAttributeId
import ru.yandex.crm.apphost.kotlin.handlers.entitymanager.repository.model.parameter.EntityTimestampAttributeParameter
import java.time.Instant
import kotlin.test.assertEquals
import kotlin.test.assertFalse
import kotlin.test.assertTrue

class TimestampAttributeValidatorTests : KoinTest {

    @Test
    fun `Should return success`() {
        val validator = TimestampAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = true,
                isNullable = false,
                timestampParameter = EntityTimestampAttributeParameter()
            )
            ),
            "test"
        )

        val value = "2022-01-02T18:00:00Z"

        val validationResult = validator.validate(value, attribute)

        assertTrue(validationResult.isValid)
    }

    @Test
    fun `Should return success when attribute is not required but value is empty`() {
        val validator = TimestampAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = false,
                isNullable = true,
                timestampParameter = EntityTimestampAttributeParameter()
            )
            ),
            "test"
        )

        val value = ""
        val validationResult = validator.validate(value, attribute)

        assertTrue(validationResult.isValid)
    }

    @Test
    fun `Should return error when attribute is required but value is empty`() {
        val validator = TimestampAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = true,
                isNullable = false,
                timestampParameter = EntityTimestampAttributeParameter()
            )
            ),
            "test"
        )

        val value = ""
        val validationResult = validator.validate(value, attribute)

        assertFalse(validationResult.isValid)
        assertEquals(ValidationResultReason.REQUIRED, validationResult.reason)
    }

    @Test
    fun `Should return error when date has incorrect format`() {
        val validator = TimestampAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = true,
                isNullable = false,
                timestampParameter = EntityTimestampAttributeParameter()
            )
            ),
            "test"
        )

        val value = "test"
        val validationResult = validator.validate(value, attribute)

        assertFalse(validationResult.isValid)
        assertEquals(ValidationResultReason.INCORRECT_FORMAT, validationResult.reason)
    }

    @Test
    fun `Should return success when attribute is required but there is default value`() {
        val validator = TimestampAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = true,
                isNullable = false,
                timestampParameter = EntityTimestampAttributeParameter(
                    defaultValue = Instant.now()
                )
            )),
            "test"
        )

        val value = ""
        val validationResult = validator.validate(value, attribute)

        assertTrue(validationResult.isValid)
    }

    @Test
    fun `Should return success when attribute is required but now is default value`() {
        val validator = TimestampAttributeValidatorImpl()

        val attribute = EntityMetaVersionAttribute(
            EntityMetaVersionAttributeId(entityAttribute = EntityAttribute(
                isRequired = true,
                isNullable = false,
                timestampParameter = EntityTimestampAttributeParameter(
                   nowIsDefault = true
                )
            )),
            "test"
        )

        val value = ""
        val validationResult = validator.validate(value, attribute)

        assertTrue(validationResult.isValid)
    }
}
