package ru.yandex.crm.apphost.kotlin.handlers.entitystorage.api

import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.api.mapper.AttributeFilterMapper
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.api.mapper.EntityMapper
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.api.mapper.EntitySchemaMapper
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.api.mapper.EntityUserMapper
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.Entity
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.EntitySchema
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.service.EntityManagementService
import ru.yandex.crm.apphost.proto.entitystorage.EntityServiceBase
import ru.yandex.crm.apphost.proto.entitystorage.Entitystorage
import ru.yandex.crm.proto.gallifrey.entitystorage.Entitystorage.EntityResponse
import ru.yandex.web.apphost.api.request.RequestMeta
import java.util.UUID

class EntityService(
    private val entityManagementService: EntityManagementService,
    private val entitySchemaMapper: EntitySchemaMapper,
    private val entityMapper: EntityMapper,
    private val attributeFilterMapper: AttributeFilterMapper,
    private val entityUserMapper: EntityUserMapper,
) : EntityServiceBase() {
    override fun create(
        requestMeta: RequestMeta,
        requestBody: Entitystorage.CreateEntityRequest
    ): Entitystorage.EntityServiceResponse {
        val entitySchema = entitySchemaMapper.fromProtobufModel(requestBody.createEntity.schema)
        val entity = entityManagementService.createEntity(
            requestBody.createEntity.organizationId,
            entitySchema,
            entityMapper.fromProtobufModel(requestBody.createEntity.entity),
            requestBody.createEntity.entity.usersList.map { entityUserMapper.fromProtobufModel(it) }
        )

        return buildServiceResponse(entity, entitySchema)
    }

    override fun update(
        requestMeta: RequestMeta,
        requestBody: Entitystorage.UpdateEntityRequest
    ): Entitystorage.EntityServiceResponse {
        val entitySchema = entitySchemaMapper.fromProtobufModel(requestBody.updateEntity.schema)
        val entity = entityManagementService.updateEntity(
            requestBody.updateEntity.organizationId,
            entitySchema,
            UUID.fromString(requestBody.updateEntity.entity.id),
            requestBody.updateEntity.entity.version,
            entityMapper.fromProtobufModelToMap(requestBody.updateEntity.entity)
        )

        return buildServiceResponse(entity, entitySchema)
    }

    override fun delete(
        requestMeta: RequestMeta,
        requestBody: Entitystorage.DeleteEntityRequest
    ): Entitystorage.EntityServiceResponse {
        val entitySchema = entitySchemaMapper.fromProtobufModel(requestBody.deleteEntity.schema)
        entityManagementService.deleteEntity(
            requestBody.deleteEntity.organizationId,
            entitySchema,
            UUID.fromString(requestBody.deleteEntity.id)
        )

        return buildServiceResponse(null, entitySchema)
    }

    override fun list(
        requestMeta: RequestMeta,
        requestBody: Entitystorage.ListEntitiesRequest
    ): Entitystorage.EntityServiceResponse {
        val result = mutableListOf<Pair<EntitySchema, Collection<Entity>>>()
        for (listCommand in requestBody.listEntitiesList) {
            val entitySchema = entitySchemaMapper.fromProtobufModel(listCommand.schema)
            val entities = entityManagementService.listEntities(
                listCommand.organizationId,
                entitySchema,
                listCommand.filtersList.map { attributeFilterMapper.fromProtobufModel(it) }
            )

            result.add(Pair(entitySchema, entities))
        }

        return buildServiceResponse(result)
    }

    private fun buildServiceResponse(entity: Entity?, schema: EntitySchema): Entitystorage.EntityServiceResponse {
        return if (entity == null) {
            buildServiceResponse(mutableListOf(), schema)
        } else {
            buildServiceResponse(mutableListOf(entity), schema)
        }
    }

    private fun buildServiceResponse(
        entities: Iterable<Entity>,
        schema: EntitySchema
    ): Entitystorage.EntityServiceResponse {
        return buildResponse(listOf(buildEntityResponse(entities, schema)))
    }

    private fun buildServiceResponse(
        entityPairs: Collection<Pair<EntitySchema, Collection<Entity>>>
    ): Entitystorage.EntityServiceResponse {
        val response = entityPairs.map {
            buildEntityResponse(it.second, it.first)
        }
        return buildResponse(response)
    }

    private fun buildResponse(
        entities: Collection<EntityResponse>
    ): Entitystorage.EntityServiceResponse {
        return Entitystorage.EntityServiceResponse.newBuilder()
            .addAllEntityResponse(entities)
            .build()
    }

    private fun buildEntityResponse(
        entities: Iterable<Entity>,
        schema: EntitySchema
    ): EntityResponse {
        return EntityResponse
            .newBuilder()
            .addAllEntities(entities.map { entityMapper.toProtobufModel(it, schema) })
            .build()
    }
}
