package ru.yandex.crm.apphost.kotlin.handlers.entitystorage.api.mapper.impl

import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.anyValue
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.api.mapper.AttributeSchemaMapper
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.api.mapper.EntityMapper
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.api.mapper.ProtoEntityRecord
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.Entity
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.EntitySchema
import ru.yandex.crm.proto.gallifrey.entitystorage.Entitystorage
import java.util.UUID

class EntityMapperImpl(
    private val attributeSchemaMapper: AttributeSchemaMapper
) : EntityMapper {
    override fun fromProtobufModelToArray(
        protoMessage: ProtoEntityRecord
    ): Array<Any?> {
        val maxFieldNumber = protoMessage.attributesList.maxOf { it.schema.fieldNumber }
        return Array(maxFieldNumber + 1) {
            val attribute = protoMessage.attributesList.first {
                    attr-> attr.schema.fieldNumber == it }

            val attributeSchema = attributeSchemaMapper.fromProtobufModel(attribute.schema)
            val attributeValue = attributeSchema.type.protoValueToValue(attribute.value.anyValue)
            attributeValue
        }
    }

    override fun fromProtobufModelToMap(
        protoMessage: ProtoEntityRecord
    ): Map<Int, Any?> {
        return protoMessage.attributesList.associate {
            val attributeSchema = attributeSchemaMapper.fromProtobufModel(it.schema)
            it.schema.fieldNumber to attributeSchema.type.protoValueToValue(it.value.anyValue)
        }
    }

    override fun toProtobufModel(model: Entity, entityMeta: EntitySchema): ProtoEntityRecord {
        val builder = ProtoEntityRecord
            .newBuilder()
            .setMetaId(model.metaId.toString())
            .setOrganizationId(model.organizationId)
            .setInternalId(model.internalId ?: -1)
            .setId(model.id.toString())

        entityMeta.attributes.forEach { attr ->
            val value = model.attributeValues[attr.fieldNumber]
            val attribute = Entitystorage.Attribute
                .newBuilder()
                .setSchema(attributeSchemaMapper.toProtobufModel(attr))
            if (value != null) {
                attribute.value = attr.type.toAttributeValue(value)
            }

            builder.addAttributes(attribute.build())
        }

        return builder.build()
    }

    override fun fromProtobufModel(protoMessage: ProtoEntityRecord): Entity =
        Entity(
            id = if (protoMessage.id.isNullOrEmpty()) null else UUID.fromString(protoMessage.id),
            internalId = protoMessage.internalId,
            metaId = UUID.fromString(protoMessage.metaId),
            organizationId = protoMessage.organizationId,
            departmentId = UUID.fromString(protoMessage.departmentId),
            version = protoMessage.version,
            attributeValues = fromProtobufModelToMap(protoMessage).toMutableMap())
}
