package ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.impl

import org.hibernate.Session
import org.hibernate.query.NativeQuery
import org.hibernate.type.StandardBasicTypes
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.EntityRepository
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.Entity
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.EntitySchema
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.filters.AttributeFilter
import java.math.BigInteger
import java.util.UUID

private const val ENTITY_SYSTEM_PARAMETERS_COUNT = 7

open class EntityRepositoryImpl(
    private val session: Session,
    private val entitySchema: EntitySchema,
) : EntityRepository {
    override fun delete(id: UUID) {
        val queryString =
            "update entitystorage.entity set is_deleted = true where id = :id"
        val query = session.createNativeQuery(queryString)
        query.setParameter("id", id)
        query.executeUpdate()
    }

    override fun findAll(organizationId: Long): List<Entity> {
        val queryString = "${generateEntitySelectClause()} from entitystorage.entity as e " +
            "where e.organization_id = :organization_id and e.meta_id = :meta_id"

        val query = session.createNativeQuery(queryString)
            .setParameter("organization_id", organizationId)
            .setParameter("meta_id", entitySchema.metaId)
            .withEntitySchema()

        return query.resultList.map { toEntity(it as Array<Any>) }
    }

    override fun findOne(id: UUID): Entity? {
        val queryString =
            "${generateEntitySelectClause()} from entitystorage.entity as e where e.id = :id"
        val query = session.createNativeQuery(queryString)
            .setParameter("id", id)
            .withEntitySchema()

        val queryResult = query.resultList

        return when (queryResult.size) {
            0 -> null
            1 -> toEntity(queryResult[0] as Array<Any>)
            else -> error("Can't be more than one entity with id $id")
        }
    }

    override fun findOne(internalId: Long): Entity? {
        val queryString =
            "${generateEntitySelectClause()} from entitystorage.entity as e where e.internal_id = :internal_id"
        val query = session.createNativeQuery(queryString)
            .setParameter("internal_id", internalId)
            .withEntitySchema()

        val queryResult = query.resultList

        return when (queryResult.size) {
            0 -> null
            1 -> toEntity(queryResult[0] as Array<Any>)
            else -> error("Can't be more than one entity with internal_id $internalId")
        }
    }

    override fun filterByInternalIdRange(
        internalIdMin: Double?,
        includeMin: Boolean?,
        internalIdMax: Double?,
        includeMax: Boolean?
    ): List<Entity> {
        var queryString =
            "${generateEntitySelectClause()} from entitystorage.entity as e where "
        var filterString = ""
        if (internalIdMin != null) {
            filterString += "and e.internal_id ${if (includeMin ?: false) ">=" else ">"} :minValue "
        }

        if (internalIdMax != null) {
            filterString += "and e.internal_id ${if (includeMax ?: false) "<=" else "<"} :maxValue "
        }

        filterString = filterString.removePrefix("and ")
        queryString += filterString
        val query = session.createNativeQuery(queryString)
        if (internalIdMin != null) {
            query.setParameter("minValue", internalIdMin)
        }
        if (internalIdMax != null) {
            query.setParameter("maxValue", internalIdMax)
        }
        query.withEntitySchema()

        return query.resultList.map { toEntity(it as Array<Any>) }
    }

    override fun filterByIdList(ids: List<UUID>): List<Entity> = listQuery(
        "${generateEntitySelectClause()} from entitystorage.entity as e where e.id in (${
            ids.indices.joinToString(",") { ":value$it" }})", ids)

    override fun filterByInternalIdsList(internalIds: List<Long>): List<Entity> = listQuery(
        "${generateEntitySelectClause()} from entitystorage.entity as e where e.internal_id in (${
            internalIds.indices.joinToString(",") { ":value$it" }})", internalIds)

    private fun listQuery(queryString: String, values: List<Any>): List<Entity> {
        val query = session.createNativeQuery(queryString)
        values.forEachIndexed { index, value -> query.setParameter("value$index", value) }
        query.withEntitySchema()
        return query.resultList.map { toEntity(it as Array<Any>) }
    }

    private fun NativeQuery<*>.withEntitySchema(): NativeQuery<*> {
        this.addScalar("id", StandardBasicTypes.UUID_BINARY)
            .addScalar("internal_id", StandardBasicTypes.BIG_INTEGER)
            .addScalar("is_deleted", StandardBasicTypes.BOOLEAN)
            .addScalar("meta_id", StandardBasicTypes.UUID_BINARY)
            .addScalar("organization_id", StandardBasicTypes.BIG_INTEGER)
            .addScalar("version", StandardBasicTypes.INTEGER)
            .addScalar("department_id", StandardBasicTypes.UUID_BINARY)



        entitySchema.attributes.forEach {
            this.addScalar("value${it.fieldNumber}", it.type.dbType)
        }

        return this
    }

    val selectionIndexMap: MutableMap<Int, Int> = mutableMapOf()

    private fun generateEntitySelectClause(): String {
        val valueNames = mutableListOf<String>()
        var selectionIndex = 0
        entitySchema.attributes.forEach {
            valueNames.add("${it.type.getCastedValueExpression("e.value${it.fieldNumber}")} as value${it.fieldNumber}")
            selectionIndexMap[it.fieldNumber] = selectionIndex
            selectionIndex++
        }
        return "select e.id, e.internal_id, e.is_deleted, e.meta_id, e.organization_id, e.version, e.department_id, " +
            valueNames.joinToString()
    }

    private fun toEntity(entity: Array<Any>): Entity {
        val maxFieldNumber = entitySchema.attributes.size - 1
        if (maxFieldNumber + ENTITY_SYSTEM_PARAMETERS_COUNT > entity.size) {
            error("Invalid entity data")
        }

        val entityBase = Entity(
            id = entity[0] as UUID,
            internalId = (entity[1] as BigInteger).toLong(),
            isDeleted = (entity[2] as Boolean?) ?: false,
            metaId = entity[3] as UUID,
            organizationId = (entity[4] as BigInteger).toLong(),
            version = (entity[5] as Integer).toInt(),
            departmentId = entity[6] as UUID,
            attributeValues = mutableMapOf()
        )

        entitySchema.attributes.forEach {
            entityBase.attributeValues[it.fieldNumber] =
                it.type.dbValueToValue(entity[selectionIndexMap[it.fieldNumber]!! + ENTITY_SYSTEM_PARAMETERS_COUNT])

        }
        return entityBase
    }

    override fun create(entity: Entity): Entity {
        val valuePairs = entitySchema.attributes.map {
            Pair("value" + it.fieldNumber, it.type.valueToDbValue(entity.attributeValues[it.fieldNumber]))
        }
        val length = valuePairs.size

        val queryString =
            "insert into entitystorage.entity (${valuePairs.joinToString { it.first }}, id, organization_id, meta_id, " +
                "is_deleted, version, department_id) values (${valuePairs.joinToString { "?" }}, ?, ?, ?, ?, ?, ?)"
        val query = session.createNativeQuery(queryString)
        valuePairs.forEachIndexed { index, pair ->
            query.setParameter(index + 1, pair.second)
        }
        entity.id = UUID.randomUUID()
        query
            .setParameter(length + 1, entity.id)
            .setParameter(length + 2, entity.organizationId.toBigInteger())
            .setParameter(length + 3, entity.metaId)
            .setParameter(length + 4, entity.isDeleted)
            .setParameter(length + 5, entity.version)
            .setParameter(length + 6, entity.departmentId)

        query.executeUpdate()

        return entity
    }

    override fun update(entityId: UUID, version: Int, updatedValues: Map<Int, Any?>): Entity {
        val valuePairs = updatedValues.map {
            Pair("value" + it.key, it.value)
        }
        val length = valuePairs.size
        val queryString =
            "update entitystorage.entity set ${valuePairs.joinToString { it.first + " = ?" }}, version = ? where id = ? and version = ?"
        val query = session.createNativeQuery(queryString)

        valuePairs.forEachIndexed { index, pair ->
            query.setParameter(index + 1, pair.second)
        }
        query.setParameter(length + 1, version + 1)
        query.setParameter(length + 2, entityId)
        query.setParameter(length + 3, version)
        val updatedCount = query.executeUpdate()

        if (updatedCount == 0) {
            error("Unable to update entity with id $entityId an version $version")
        }
        return findOne(entityId) ?: error("Can't find updated entity")
    }

    override fun filter(organizationId: Long, attributeFilter: AttributeFilter): List<Entity> {
        val queryString = "${generateEntitySelectClause()} from entitystorage.index as idx " +
            "left join entitystorage.entity as e " +
            "on idx.entity_id = e.id " +
            "where idx.organization_id = :organization_id and idx.meta_id = :meta_id and idx.field_number = :field_number " +
            attributeFilter.filter.getQueryString(attributeFilter.schema.type)

        val query = session.createNativeQuery(queryString)
            .setParameter("organization_id", organizationId)
            .setParameter("meta_id", entitySchema.metaId)
            .setParameter("field_number", attributeFilter.schema.fieldNumber)
            .withEntitySchema()

        for (parameter in attributeFilter.filter.getParameterList(attributeFilter.schema.type)) {
            query.setParameter(parameter.key, parameter.value)
        }

        return query.resultList.map { toEntity(it as Array<Any>) }
    }
}
