package ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model

import org.hibernate.type.StandardBasicTypes
import ru.yandex.crm.apphost.kotlin.common.extensions.toInstant
import ru.yandex.crm.apphost.kotlin.common.extensions.toProtobufTimestamp
import java.time.Instant
import org.hibernate.type.Type
import ru.yandex.crm.proto.gallifrey.entitystorage.Entitystorage.AttributeValue
import java.sql.Timestamp

typealias ProtoTimestamp = com.google.protobuf.Timestamp

sealed interface AttributeType {
    val queryIndexField: String
    val dbType: Type

    fun getCastedValueExpression(entityColumnName: String): String
    fun valueFromString(value: String): Any
    fun protoValueToValue(value: Any): Any
    fun dbValueToValue(value: Any?): Any?
    fun valueToDbValue(value: Any?): Any?
    fun toAttributeValue(value: Any): AttributeValue
    fun setValueToIndex(index: Index, value: Any?)
}

sealed class AttributeTypeImpl<ModelType : Any, ProtoType : Any, DbMappingType: Any> : AttributeType {
    object STRING : AttributeTypeImpl<String, String, String>() {
        override val queryIndexField: String
            get() = "string_value"

        override fun valueFromStringImpl(value: String): String = value

        override val dbType: Type
            get() = StandardBasicTypes.TEXT

        override fun setValueToIndexImpl(index: Index, value: String?) {
            index.stringValue = value
        }

        override fun toAttributeValueImpl(value: String): AttributeValue =
            AttributeValue.newBuilder().setStringValue(valueToProtoValueImpl(value)).build()

        override fun getCastedValueExpression(entityColumnName: String): String = entityColumnName
        override fun dbValueToValueImpl(value: String?): String? = value
        override fun valueToDbValueImpl(value: String?): String? = value
        override fun protoValueToValueImpl(value: String): String = value
        override fun valueToProtoValueImpl(value: String): String = value
    }

    object INT : AttributeTypeImpl<Int, Int, Integer>() {
        override val queryIndexField: String
            get() = "int_value"

        override fun valueFromStringImpl(value: String): Int = value.toInt()

        override val dbType: Type
            get() = StandardBasicTypes.INTEGER

        override fun setValueToIndexImpl(index: Index, value: Int?) {
            index.intValue = value
        }

        override fun toAttributeValueImpl(value: Int): AttributeValue =
            AttributeValue.newBuilder().setIntValue(valueToProtoValueImpl(value)).build()

        override fun dbValueToValueImpl(value: Integer?): Int? = value?.toInt()
        override fun valueToDbValueImpl(value: Int?): Integer? = value as Integer?
        override fun protoValueToValueImpl(value: Int): Int = value
        override fun valueToProtoValueImpl(value: Int): Int = value
    }

    object TIMESTAMP : AttributeTypeImpl<Instant, ProtoTimestamp, Timestamp>() {
        override val queryIndexField: String
            get() = "timestamp_value"

        override val dbType: Type
            get() = StandardBasicTypes.TIMESTAMP

        override fun setValueToIndexImpl(index: Index, value: Instant?) {
            index.timestampValue = value
        }

        override fun toAttributeValueImpl(value: Instant): AttributeValue =
            AttributeValue.newBuilder().setTimestampValue(valueToProtoValueImpl(value)).build()

        override fun valueFromStringImpl(value: String): Instant  = Instant.parse(value)

        override fun dbValueToValueImpl(value: Timestamp?): Instant? = value?.toInstant()
        override fun valueToDbValueImpl(value: Instant?): Timestamp? =
            if (value !=null) Timestamp.from(value) else null
        override fun protoValueToValueImpl(value: ProtoTimestamp): Instant = value.toInstant()
        override fun valueToProtoValueImpl(value: Instant): ProtoTimestamp = value.toProtobufTimestamp()
    }

    object BOOL: AttributeTypeImpl<Boolean, Boolean, Boolean>(){
        override val queryIndexField: String
            get() = "bool_value"

        override val dbType: Type
            get() = StandardBasicTypes.BOOLEAN

        override fun setValueToIndexImpl(index: Index, value: Boolean?) {
            index.boolValue = value
        }

        override fun toAttributeValueImpl(value: Boolean): AttributeValue =
            AttributeValue.newBuilder().setBoolValue(valueToProtoValueImpl(value)).build()

        override fun valueFromStringImpl(value: String): Boolean  = value.toBoolean()
        override fun dbValueToValueImpl(value: Boolean?): Boolean? = value
        override fun valueToDbValueImpl(value: Boolean?): Boolean? = value
        override fun protoValueToValueImpl(value: Boolean): Boolean = value
        override fun valueToProtoValueImpl(value: Boolean): Boolean = value
    }

    object UUID: AttributeTypeImpl<java.util.UUID, String, java.util.UUID>(){
        override val queryIndexField: String
            get() = "uuid_value"

        override val dbType: Type
            get() = StandardBasicTypes.UUID_BINARY

        override fun setValueToIndexImpl(index: Index, value: java.util.UUID?) {
            index.uuidValue = value
        }

        override fun toAttributeValueImpl(value: java.util.UUID): AttributeValue =
            AttributeValue.newBuilder().setUuidValue(valueToProtoValueImpl(value)).build()

        override fun getCastedValueExpression(entityColumnName: String): String = "CAST ($entityColumnName AS uuid)"

        override fun valueFromStringImpl(value: String): java.util.UUID  = java.util.UUID.fromString(value)
        override fun dbValueToValueImpl(value: java.util.UUID?): java.util.UUID? = value
        override fun valueToDbValueImpl(value: java.util.UUID?): java.util.UUID? = value
        override fun protoValueToValueImpl(value: String): java.util.UUID = java.util.UUID.fromString(value)
        override fun valueToProtoValueImpl(value: java.util.UUID): String = value.toString()
    }

    object FLOAT: AttributeTypeImpl<Float, Float, java.lang.Float>(){
        override val queryIndexField: String
            get() = "float_value"

        override val dbType: Type
            get() = StandardBasicTypes.FLOAT

        override fun setValueToIndexImpl(index: Index, value: Float?) {
            index.floatValue = value
        }

        override fun toAttributeValueImpl(value: Float): AttributeValue =
            AttributeValue.newBuilder().setFloatValue(valueToProtoValueImpl(value)).build()

        override fun getCastedValueExpression(entityColumnName: String): String = "CAST ($entityColumnName AS float)"

        override fun valueFromStringImpl(value: String): Float  = value.toFloat()
        override fun dbValueToValueImpl(value: java.lang.Float?): Float? = value?.toFloat()
        override fun valueToDbValueImpl(value: Float?): java.lang.Float? = value as java.lang.Float?
        override fun protoValueToValueImpl(value: Float): Float = value
        override fun valueToProtoValueImpl(value: Float): Float = value
    }

    object DOUBLE: AttributeTypeImpl<Double, Double, java.lang.Double>(){
        override val queryIndexField: String
            get() = "double_value"

        override val dbType: Type
            get() = StandardBasicTypes.DOUBLE

        override fun setValueToIndexImpl(index: Index, value: Double?) {
            index.doubleValue = value
        }

        override fun toAttributeValueImpl(value: Double): AttributeValue =
            AttributeValue.newBuilder().setDoubleValue(valueToProtoValueImpl(value)).build()

        override fun getCastedValueExpression(entityColumnName: String): String = "CAST ($entityColumnName AS double)"

        override fun valueFromStringImpl(value: String): Double  = value.toDouble()
        override fun dbValueToValueImpl(value: java.lang.Double?): Double? = value?.toDouble()
        override fun valueToDbValueImpl(value: Double?): java.lang.Double? = value as java.lang.Double?
        override fun protoValueToValueImpl(value: Double): Double = value
        override fun valueToProtoValueImpl(value: Double): Double = value
    }

    override fun getCastedValueExpression(entityColumnName: String): String =
        "CAST ($entityColumnName AS ${dbType.name})"

    override fun valueFromString(value: String): Any = valueFromStringImpl(value)
    protected abstract fun valueFromStringImpl(value: String): ModelType

    protected abstract fun valueToProtoValueImpl(value: ModelType): ProtoType

    override fun protoValueToValue(value: Any): Any = protoValueToValueImpl(value as ProtoType)
    protected abstract fun protoValueToValueImpl(value: ProtoType): ModelType

    override fun toAttributeValue(value: Any) = toAttributeValueImpl(value as ModelType)
    protected abstract fun toAttributeValueImpl(value: ModelType): AttributeValue

    override fun setValueToIndex(index: Index, value: Any?) = setValueToIndexImpl(index, value as ModelType?)
    protected abstract fun setValueToIndexImpl(index: Index, value: ModelType?)

    override fun dbValueToValue(value: Any?): Any? =  dbValueToValueImpl(value as DbMappingType?)
    protected abstract fun dbValueToValueImpl(value: DbMappingType?): ModelType?

    override fun valueToDbValue(value: Any?): Any? =  valueToDbValueImpl(value as ModelType?)
    protected abstract fun valueToDbValueImpl(value: ModelType?): DbMappingType?
}
