package ru.yandex.crm.apphost.kotlin.handlers.entitystorage.service.impl

import mu.KotlinLogging
import org.hibernate.Session
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.EntityRepository
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.EntityUserRepository
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.IndexRepository
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.getEntityRepository
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.Entity
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.EntitySchema
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.EntityUser
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.EntityUserRoleEnum
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.Index
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.filters.AttributeFilter
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.filters.impl.EqualityFilter
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.filters.impl.NumberRangeFilter
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.filters.impl.ValueListFilter
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.service.EntityManagementService
import ru.yandex.crm.library.kotlin.database.hibernate.getRepository
import ru.yandex.crm.library.kotlin.database.hibernate.transaction
import java.util.UUID

class EntityManagementServiceImpl : EntityManagementService {

    override fun createEntity(
        organizationId: Long,
        entitySchema: EntitySchema,
        entity: Entity,
        users: List<EntityUser>
    ): Entity {

        val authorsCount = users.count { it.role == EntityUserRoleEnum.AUTHOR }
        if (authorsCount == 0)
            error("Unable to create entity without author")

        if (authorsCount > 1)
            error("Unable to create entity with multiple author")

        return transaction {
            val entityRepository = getEntityRepository(entitySchema)
            val intermediateEntity = entityRepository.create(entity)
            val createdEntity = entityRepository.findOne(intermediateEntity.id!!)
                ?: error("Invalid creation operation")

            val entityUserRepository = getRepository<EntityUserRepository>()
            users.forEach {
                it.entityId = createdEntity.id
                entityUserRepository.save(it)
            }

            saveIndexes(entitySchema, createdEntity)
            createdEntity
        }
    }

    override fun updateEntity(
        organizationId: Long,
        entitySchema: EntitySchema,
        entityId: UUID,
        version: Int,
        attributes: Map<Int, Any?>
    ): Entity {
        return transaction {
            val entityRepository = getEntityRepository(entitySchema)
            val entity = entityRepository.update(entityId, version, attributes)
            saveIndexes(entitySchema, entity)
            entity
        }
    }

    private fun Session.saveIndexes(
        entitySchema: EntitySchema,
        entity: Entity
    ) {
        val indexRepository = getRepository<IndexRepository>()
        entitySchema.attributes.filter { it.isIndexed }.forEach {
            val entityIndex = Index(entitySchema.organizationId, entitySchema.metaId, entity.id, it.fieldNumber)
            it.type.setValueToIndex(entityIndex, entity.attributeValues[it.fieldNumber])
            indexRepository.save(entityIndex)
        }
    }

    override fun deleteEntity(
        organizationId: Long,
        entitySchema: EntitySchema,
        entityId: UUID
    ) {
        return transaction {
            val entityRepository = getEntityRepository(entitySchema)
            entityRepository.delete(entityId)
        }
    }

    override fun listEntities(
        organizationId: Long,
        entitySchema: EntitySchema,
        filters: List<AttributeFilter>
    ): List<Entity> = transaction {
            val entityRepository = getEntityRepository(entitySchema)
            if (filters.any()) {
                val firstFilter = filters[0]
                return when (firstFilter.schema.name) {
                    ID_NAME -> filterById(entityRepository, firstFilter)
                    INTERNAL_ID_NAME -> filterByInternalId(entityRepository,firstFilter)
                    else -> entityRepository.filter(organizationId, firstFilter)
                }
            } else {
                return entityRepository.findAll(organizationId)
            }
        }

    private fun filterById(entityRepository: EntityRepository, filter: AttributeFilter): List<Entity> =
        when (filter.filter) {
            is EqualityFilter -> {
                val entity = entityRepository.findOne(filter.filter.value as UUID)
                if (entity == null) listOf() else listOf(entity)
            }
            is ValueListFilter -> {
                entityRepository.filterByIdList(filter.filter.valueList.map { it as UUID })
            }
            else -> {
                error("Only equality or list filter can be used with id attribute")
            }
        }

    private fun filterByInternalId(entityRepository: EntityRepository, filter: AttributeFilter): List<Entity> =
        when (filter.filter) {
            is EqualityFilter -> {
                val entity = entityRepository.findOne((filter.filter.value as Integer).toLong())
                if (entity == null) listOf() else listOf(entity)
            }
            is NumberRangeFilter -> {
                entityRepository.filterByInternalIdRange(
                    filter.filter.minValue, filter.filter.includeMin, filter.filter.maxValue, filter.filter.includeMax)
            }
            is ValueListFilter -> {
                entityRepository.filterByInternalIdsList(filter.filter.valueList.map { (it as Integer).toLong() })
            }
            else -> {
                error("Only equality, range or list filter can be used with internal_id attribute")
            }
        }

    companion object {
        const val ID_NAME = "id"
        const val INTERNAL_ID_NAME = "internal_id"
        private val logger = KotlinLogging.logger { }
    }
}
