package ru.yandex.crm.apphost.kotlin.handlers.entitystorage.ut

import org.apache.commons.dbcp2.BasicDataSource
import org.junit.jupiter.api.AfterEach
import org.junit.jupiter.api.BeforeEach
import org.koin.core.context.startKoin
import org.koin.core.context.stopKoin
import org.koin.dsl.bind
import org.koin.dsl.module
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.EntityRepository
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.EntityUserRepository
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.IndexRepository
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.MetaSchema
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.SchemalessEntityRepository
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.impl.EntityRepositoryImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.impl.EntityUserRepositoryImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.impl.IndexRepositoryImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.impl.SchemalessEntityRepositoryImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.Entity
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.EntitySchema
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.service.EntityManagementService
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.service.EntityUserManager
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.service.impl.EntityManagementServiceImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.service.impl.EntityUserManagerImpl
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateConfig
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateSchema
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateSessionManager
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateSessionManagerImpl
import ru.yandex.crm.library.kotlin.database.hibernate.transaction
import java.time.Instant
import java.util.UUID
import javax.sql.DataSource

abstract class EntityDbBackedTests : EntityTests() {
    protected open val hibernateConfig = HibernateConfig().apply {
        dialect = "org.hibernate.dialect.H2Dialect"
        showSql = true
        hbm2ddlAuto = "update"
    }

    protected open val dataSource = BasicDataSource().apply {
        driverClassName = "org.h2.Driver"
        url = "jdbc:h2:mem:myDb;DB_CLOSE_DELAY=-1;INIT=CREATE SCHEMA IF NOT EXISTS ENTITYSTORAGE"
    }

    private val entityStorageModule = module {
        single { hibernateConfig } bind HibernateConfig::class
        single { dataSource } bind DataSource::class
        single { MetaSchema() } bind HibernateSchema::class
        single { HibernateSessionManagerImpl(get(), get(), getAll()) } bind HibernateSessionManager::class
        single { EntityManagementServiceImpl() } bind EntityManagementService::class
        single { EntityUserManagerImpl() } bind EntityUserManager::class

        factory { params -> EntityRepositoryImpl(params.get(), params.get()) } bind EntityRepository::class
        factory { params -> SchemalessEntityRepositoryImpl(params.get()) } bind SchemalessEntityRepository::class
        factory { params -> IndexRepositoryImpl(params.get()) } bind IndexRepository::class
        factory { params -> EntityUserRepositoryImpl(params.get()) } bind EntityUserRepository::class
    }

    @BeforeEach
    fun beforeEachBase() {
        startKoin {
            modules(
                entityStorageModule
            )
        }
        transaction {
            createNativeQuery(
                "create table entitystorage.entity (id binary not null, internal_id serial, is_deleted boolean, " +
                    "meta_id binary not null, organization_id bigint not null, version integer not null, department_id binary not null, value0 varchar(255), " +
                    "value1 varchar(255), value2 varchar(255), value3 varchar(255), value4 varchar(255), value5 varchar(255), value6 varchar(255), value7 varchar(255), value8 varchar(255), value9 varchar(255), " +
                    "primary key (internal_id))"
            )
                .executeUpdate()
        }
    }

    @AfterEach
    fun afterEachBase() {
        transaction {
            createNativeQuery("DROP ALL OBJECTS").executeUpdate()
        }
        stopKoin()
    }

    protected fun createEntity(
        entitySchema: EntitySchema,
        dt: Instant,
        author: String = "Pushkin",
        severity: Int = 50,
        active: Boolean = false,
        uuid: UUID = UUID.fromString("973545e4-f09d-11ec-8ea0-0242ac120002"),
        float:Float = 3.1415f,
        double: Double = 2.7182818284590,
        departmentId: UUID? = null
    ): Entity {
        return Entity(
            organizationId = 1,
            metaId = entitySchema.metaId,
            version = 1,
            departmentId = departmentId ?: UUID.randomUUID(),
            attributeValues = mutableMapOf(
                0 to author,
                1 to severity,
                2 to dt,
                3 to active,
                4 to uuid,
                5 to float,
                6 to double
            )
        )
    }
}
