package ru.yandex.crm.apphost.kotlin.handlers.entitystorage.ut.mapper

import org.junit.jupiter.api.BeforeEach
import org.junit.jupiter.api.Test
import org.junit.jupiter.api.assertThrows
import ru.yandex.crm.apphost.kotlin.common.extensions.toProtobufTimestamp
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.anyValue
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.api.mapper.EntityMapper
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.api.mapper.ProtoEntitySchema
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.api.mapper.impl.AttributeSchemaMapperImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.api.mapper.impl.AttributeTypeMapperImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.api.mapper.impl.EntityMapperImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.api.mapper.impl.EntitySchemaMapperImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.Entity
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.EntitySchema
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.ut.EntityTests
import ru.yandex.crm.proto.gallifrey.entitystorage.Entitystorage.Attribute
import ru.yandex.crm.proto.gallifrey.entitystorage.Entitystorage.AttributeSchema
import ru.yandex.crm.proto.gallifrey.entitystorage.Entitystorage.AttributeValue
import ru.yandex.crm.proto.gallifrey.entitystorage.Entitystorage.EntityRecord
import java.time.Instant
import java.time.temporal.ChronoUnit
import java.util.UUID
import kotlin.test.assertContentEquals
import kotlin.test.assertEquals
import kotlin.test.assertNotNull

open class EntityMapperTests : EntityTests() {

    private lateinit var entitySchema: EntitySchema
    private lateinit var protoEntitySchema: ProtoEntitySchema
    private lateinit var entityMapper: EntityMapper

    @BeforeEach
    fun beforeEach() {
        entitySchema = createEntitySchema()
        protoEntitySchema = EntitySchemaMapperImpl(AttributeSchemaMapperImpl(AttributeTypeMapperImpl())).toProtobufModel(entitySchema)
        entityMapper = EntityMapperImpl(
            AttributeSchemaMapperImpl(AttributeTypeMapperImpl()))
    }

    @Test
    fun `create Entity model with attributes and map to proto`() {
        //Arrange
        val dt = Instant.now().truncatedTo(ChronoUnit.MILLIS)
        val uuid = UUID.fromString("973545e4-f09d-11ec-8ea0-0242ac120002")
        val entityId = UUID.randomUUID()
        val metaId = entitySchema.metaId

        val entity = Entity(
            organizationId = 1,
            metaId = metaId,
            version = 1,
            attributeValues = mutableMapOf(
                0 to "New value",
                1 to 50,
                2 to dt,
                3 to true,
                4 to uuid,
                5 to 3.1415f,
                6 to 2.7182818284590
            )
        ).apply {
            id = entityId
        }

        //Act
        val protoEntity = entityMapper.toProtobufModel(entity, entitySchema)

        //Assert
        assertNotNull(protoEntity)
        assertEquals(entityId.toString(), protoEntity.id)
        assertEquals(entity.organizationId, protoEntity.organizationId)
        assertEquals(protoEntity.metaId.toString(), protoEntitySchema.metaId)
        assertEquals(7, protoEntity.attributesList.size)
        assertEquals("New value", protoEntity.attributesList.first { it.schema.name == "Author" }.value.anyValue)
        assertEquals(50, protoEntity.attributesList.first { it.schema.name == "Severity" }.value.anyValue)
        assertEquals(dt.toProtobufTimestamp(), protoEntity.attributesList.first { it.schema.name == "Created_at" }.value.anyValue)
        assertEquals(true, protoEntity.attributesList.first { it.schema.name == "Active" }.value.anyValue)
        assertEquals("973545e4-f09d-11ec-8ea0-0242ac120002", protoEntity.attributesList.first { it.schema.name == "Uuid" }.value.anyValue)
        assertEquals(3.1415f, protoEntity.attributesList.first { it.schema.name == "Float" }.value.anyValue)
        assertEquals(2.7182818284590, protoEntity.attributesList.first { it.schema.name == "Double" }.value.anyValue)
    }

    @Test
    fun `throw exception when invalid int value`() {
        //Arrange
        val dt = Instant.now().truncatedTo(ChronoUnit.MILLIS)
        val entityId = UUID.randomUUID()
        val metaId = entitySchema.metaId

        val entity = Entity(
            organizationId = 1,
            metaId = metaId,
            version = 1,
            attributeValues = mutableMapOf(
                0 to "New value",
                1 to "BAD INT",
                2 to dt.toString()
            )
        ).apply {
            id = entityId
        }

        //Assert
        assertThrows<Exception> { entityMapper.toProtobufModel(entity, entitySchema) }
    }

    @Test
    fun `throw exception when invalid timestamp value`() {
        //Arrange
        val entityId = UUID.randomUUID()
        val metaId = entitySchema.metaId

        val entity = Entity(
            organizationId = 1,
            metaId = metaId,
            version = 1,
            attributeValues = mutableMapOf(
                0 to "New value",
                1 to 50,
                2 to "BAD TIMESTAMP"
            )
        ).apply {
            id = entityId
        }

        //Assert
        assertThrows<Exception> { entityMapper.toProtobufModel(entity, entitySchema)}
    }

    @Test
    fun `correctly map entity proto to array of values`() {
        //Arrange
        val entityId = UUID.randomUUID()
        val metaId = entitySchema.metaId
        val dt = Instant.now().truncatedTo(ChronoUnit.MILLIS)

        val protoEntity = EntityRecord.newBuilder()
            .setId(entityId.toString())
            .setOrganizationId(1)
            .setMetaId(metaId.toString())
            .addAttributes(
                Attribute.newBuilder()
                    .setSchema(
                        AttributeSchema.newBuilder()
                            .setType(AttributeSchema.AttributeType.STRING)
                            .setName("Author")
                            .setFieldNumber(0)
                            .build()
                    )
                    .setValue(AttributeValue.newBuilder().setStringValue("New value").build())
                    .build()
            )
            .addAttributes(
                Attribute.newBuilder()
                    .setSchema(
                        AttributeSchema.newBuilder()
                            .setType(AttributeSchema.AttributeType.INT)
                            .setName("Severity")
                            .setFieldNumber(1)
                            .build()
                    )
                    .setValue(AttributeValue.newBuilder().setIntValue(50).build())
                    .build()
            )
            .addAttributes(
                Attribute.newBuilder()
                    .setSchema(
                        AttributeSchema.newBuilder()
                            .setType(AttributeSchema.AttributeType.TIMESTAMP)
                            .setName("Created_at")
                            .setFieldNumber(2)
                            .build()
                    )
                    .setValue(AttributeValue.newBuilder().setTimestampValue(dt.toProtobufTimestamp()).build())
                    .build()
            )
            .build()
        //Act
        val attributes = entityMapper.fromProtobufModelToArray(protoEntity)

        //Assert
        assertEquals(3, attributes.size)
        assertContentEquals(arrayOf("New value", 50, dt), attributes)
    }

    @Test
    fun `map proto message to array with incorrect int value`() {
        //Arrange
        val entityId = UUID.randomUUID()
        val metaId = entitySchema.metaId
        val dt = Instant.now().truncatedTo(ChronoUnit.MILLIS)

        val protoEntity = EntityRecord.newBuilder()
            .setId(entityId.toString())
            .setMetaId(metaId.toString())
            .setOrganizationId(1)
            .addAttributes(
                Attribute.newBuilder()
                    .setSchema(
                        AttributeSchema.newBuilder()
                            .setType(AttributeSchema.AttributeType.STRING)
                            .setName("Author")
                            .setFieldNumber(0)
                            .build()
                    )
                    .setValue(AttributeValue.newBuilder().setStringValue("New value").build())
                    .build()
            )
            .addAttributes(
                Attribute.newBuilder()
                    .setSchema(
                        AttributeSchema.newBuilder()
                            .setType(AttributeSchema.AttributeType.INT)
                            .setName("Severity")
                            .setFieldNumber(1)
                            .build()
                    )
                    .setValue(AttributeValue.newBuilder().setStringValue("BAD INT").build())
                    .build()
            )
            .addAttributes(
                Attribute.newBuilder()
                    .setSchema(
                        AttributeSchema.newBuilder()
                            .setType(AttributeSchema.AttributeType.TIMESTAMP)
                            .setName("Created_at")
                            .setFieldNumber(2)
                            .build()
                    )
                    .setValue(AttributeValue.newBuilder().setTimestampValue(dt.toProtobufTimestamp()).build())
                    .build()
            )
            .build()
        //Act
        assertThrows<Exception> { entityMapper.fromProtobufModelToArray(protoEntity) }
    }

    @Test
    fun `map proto message to array with incorrect timestamp value`() {
        //Arrange
        val entityId = UUID.randomUUID()
        val metaId = entitySchema.metaId

        val protoEntity = EntityRecord.newBuilder()
            .setId(entityId.toString())
            .setMetaId(metaId.toString())
            .setOrganizationId(1)
            .addAttributes(
                Attribute.newBuilder()
                    .setSchema(
                        AttributeSchema.newBuilder()
                            .setType(AttributeSchema.AttributeType.STRING)
                            .setName("Author")
                            .setFieldNumber(0)
                            .build()
                    )
                    .setValue(AttributeValue.newBuilder().setStringValue("New value").build())
                    .build()
            )
            .addAttributes(
                Attribute.newBuilder()
                    .setSchema(
                        AttributeSchema.newBuilder()
                            .setType(AttributeSchema.AttributeType.INT)
                            .setName("Severity")
                            .setFieldNumber(1)
                            .build()
                    )
                    .setValue(AttributeValue.newBuilder().setIntValue(50).build())
                    .build()
            )
            .addAttributes(
                Attribute.newBuilder()
                    .setSchema(
                        AttributeSchema.newBuilder()
                            .setType(AttributeSchema.AttributeType.TIMESTAMP)
                            .setName("Created_at")
                            .setFieldNumber(2)
                            .build()
                    )
                    .setValue(AttributeValue.newBuilder().setStringValue("BAD TIMESTAMP").build())
                    .build()
            )
            .build()
        //Act
        assertThrows<Exception> { entityMapper.fromProtobufModelToArray(protoEntity) }
    }

    @Test
    fun `correctly map entity proto to map of values`() {
        //Arrange
        val entityId = UUID.randomUUID()
        val metaId = entitySchema.metaId
        val dt = Instant.now().truncatedTo(ChronoUnit.MILLIS)

        val protoEntity = EntityRecord.newBuilder()
            .setId(entityId.toString())
            .setMetaId(metaId.toString())
            .setOrganizationId(1)
            .addAttributes(
                Attribute.newBuilder()
                    .setSchema(
                        AttributeSchema.newBuilder()
                            .setType(AttributeSchema.AttributeType.INT)
                            .setName("Severity")
                            .setFieldNumber(1)
                            .build()
                    )
                    .setValue(AttributeValue.newBuilder().setIntValue(50).build())
                    .build()
            )
            .addAttributes(
                Attribute.newBuilder()
                    .setSchema(
                        AttributeSchema.newBuilder()
                            .setType(AttributeSchema.AttributeType.TIMESTAMP)
                            .setName("Created_at")
                            .setFieldNumber(2)
                            .build()
                    )
                    .setValue(AttributeValue.newBuilder().setTimestampValue(dt.toProtobufTimestamp()).build())
                    .build()
            )
            .build()
        //Act
        val attributes = entityMapper.fromProtobufModelToMap(protoEntity)

        //Assert
        assertEquals(2, attributes.size)
        assertEquals(50, attributes[1])
        assertEquals(dt, attributes[2])
    }

    @Test
    fun `map entity proto with incorrect int to map of values`() {
        //Arrange
        val entityId = UUID.randomUUID()
        val metaId = entitySchema.metaId
        val dt = Instant.now().truncatedTo(ChronoUnit.MILLIS)

        val protoEntity = EntityRecord.newBuilder()
            .setId(entityId.toString())
            .setMetaId(metaId.toString())
            .setOrganizationId(1)
            .addAttributes(
                Attribute.newBuilder()
                    .setSchema(
                        AttributeSchema.newBuilder()
                            .setType(AttributeSchema.AttributeType.INT)
                            .setName("Severity")
                            .setFieldNumber(1)
                            .build()
                    )
                    .setValue(AttributeValue.newBuilder().setStringValue("BAD INT").build())
                    .build()
            )
            .addAttributes(
                Attribute.newBuilder()
                    .setSchema(
                        AttributeSchema.newBuilder()
                            .setType(AttributeSchema.AttributeType.TIMESTAMP)
                            .setName("Created_at")
                            .setFieldNumber(2)
                            .build()
                    )
                    .setValue(AttributeValue.newBuilder().setTimestampValue(dt.toProtobufTimestamp()).build())
                    .build()
            )
            .build()
        //Act
        assertThrows<Exception> { entityMapper.fromProtobufModelToMap(protoEntity) }
    }

    @Test
    fun `map entity proto with incorrect timestamp to map of values`() {
        //Arrange
        val entityId = UUID.randomUUID()
        val metaId = entitySchema.metaId

        val protoEntity = EntityRecord.newBuilder()
            .setId(entityId.toString())
            .setMetaId(metaId.toString())
            .setOrganizationId(1)
            .addAttributes(
                Attribute.newBuilder()
                    .setSchema(
                        AttributeSchema.newBuilder()
                            .setType(AttributeSchema.AttributeType.INT)
                            .setName("Severity")
                            .setFieldNumber(1)
                            .build()
                    )
                    .setValue(AttributeValue.newBuilder().setIntValue(50).build())
                    .build()
            )
            .addAttributes(
                Attribute.newBuilder()
                    .setSchema(
                        AttributeSchema.newBuilder()
                            .setType(AttributeSchema.AttributeType.TIMESTAMP)
                            .setName("Created_at")
                            .setFieldNumber(2)
                            .build()
                    )
                    .setValue(AttributeValue.newBuilder().setStringValue("BAD TIMESTAMP").build())
                    .build()
            )
            .build()
        //Act
        assertThrows<Exception> { entityMapper.fromProtobufModelToMap(protoEntity) }
    }
}
