package ru.yandex.crm.apphost.kotlin.handlers.entitystorage.ut.service

import org.junit.jupiter.api.Assertions.assertEquals
import org.junit.jupiter.api.Assertions.assertFalse
import org.junit.jupiter.api.Assertions.assertTrue
import org.junit.jupiter.api.BeforeEach
import org.junit.jupiter.api.Test
import org.junit.jupiter.api.assertThrows
import org.koin.test.get
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.AttributeSchema
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.AttributeTypeImpl
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.EntitySchema
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.EntityUser
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.EntityUserRoleEnum
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.filters.AttributeFilter
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.filters.impl.EqualityFilter
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.filters.impl.NumberRangeFilter
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.filters.impl.ValueListFilter
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.service.EntityManagementService
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.ut.EntityDbBackedTests
import java.time.Instant
import java.time.temporal.ChronoUnit
import java.util.UUID

class AttributeFilterTests : EntityDbBackedTests() {

    lateinit var entitySchema: EntitySchema

    @BeforeEach
    fun beforeEach() {
        entitySchema = createEntitySchema()
    }

    @Test
    fun `list all entities with equality filter on dynamic attribute`() {
        // Arrange
        val entityManagementService = get<EntityManagementService>()
        (1..5).forEach {
            val dt = Instant.now().truncatedTo(ChronoUnit.MILLIS)
            val entity = createEntity(entitySchema,dt,"Author$it", it, false, UUID.randomUUID(), it.toFloat(), it.toDouble())
            val users = listOf(EntityUser(userId = 1, role = EntityUserRoleEnum.AUTHOR))
            get<EntityManagementService>().createEntity(1, entitySchema, entity, users)
        }
        // Act
        val filters = listOf(
            AttributeFilter(
                schema = AttributeSchema(
                    type = AttributeTypeImpl.STRING,
                    fieldNumber = 0,
                    name = "Author",
                    isIndexed = true
                ),
                filter = EqualityFilter(value = "Author2")
            )
        )
        val entities = entityManagementService.listEntities(1, entitySchema, filters)

        // Assert
        assertEquals(1, entities.size)
        assertEquals("Author2", entities[0].attributeValues[0])
        assertEquals(2, entities[0].attributeValues[1])
    }

    @Test
    fun `list all entities with equality filter on id`() {
        // Arrange
        val entityManagementService = get<EntityManagementService>()
        var lastUUID = UUID.randomUUID()
        (1..5).forEach {
            val dt = Instant.now().truncatedTo(ChronoUnit.MILLIS)
            val entity = createEntity(entitySchema,dt,"Author$it", it, false, UUID.randomUUID(), it.toFloat(), it.toDouble())
            val users = listOf(EntityUser(userId = 1, role = EntityUserRoleEnum.AUTHOR))
            val outEntity = get<EntityManagementService>().createEntity(1, entitySchema, entity, users)
            lastUUID = outEntity.id
        }
        // Act
        val filters = listOf(
            AttributeFilter(
                schema = AttributeSchema(
                    type = AttributeTypeImpl.STRING,
                    fieldNumber = 123,
                    name = "id",
                    isIndexed = false
                ),
                filter = EqualityFilter(value = lastUUID)
            )
        )
        val entities = entityManagementService.listEntities(1, entitySchema, filters)

        // Assert
        assertEquals(1, entities.size)
        assertEquals("Author5", entities[0].attributeValues[0])
        assertEquals(5, entities[0].attributeValues[1])
    }

    @Test
    fun `list  entities with equality filter on internal_id`() {
        // Arrange
        val entityManagementService = get<EntityManagementService>()
        var lastUUID = UUID.randomUUID()
        (1..5).forEach {
            val dt = Instant.now().truncatedTo(ChronoUnit.MILLIS)
            val entity = createEntity(entitySchema,dt,"Author$it", it, false, UUID.randomUUID(), it.toFloat(), it.toDouble())
            val users = listOf(EntityUser(userId = 1, role = EntityUserRoleEnum.AUTHOR))
            val outEntity = get<EntityManagementService>().createEntity(1, entitySchema, entity, users)
            lastUUID = outEntity.id
        }
        // Act
        val filters = listOf(
            AttributeFilter(
                schema = AttributeSchema(
                    type = AttributeTypeImpl.INT,
                    fieldNumber = -1,
                    name = "internal_id",
                    isIndexed = false
                ),
                filter = EqualityFilter(value = 3)
            )
        )
        val entities = entityManagementService.listEntities(1, entitySchema, filters)

        // Assert
        assertEquals(1, entities.size)
        assertEquals("Author3", entities[0].attributeValues[0])
        assertEquals(3, entities[0].attributeValues[1])
    }

    @Test
    fun `list  entities with range filter on internal_id`() {
        // Arrange
        val entityManagementService = get<EntityManagementService>()
        var lastUUID = UUID.randomUUID()
        (1..5).forEach {
            val dt = Instant.now().truncatedTo(ChronoUnit.MILLIS)
            val entity = createEntity(entitySchema,dt,"Author$it", it, false, UUID.randomUUID(), it.toFloat(), it.toDouble())
            val users = listOf(EntityUser(userId = 1, role = EntityUserRoleEnum.AUTHOR))
            val outEntity = get<EntityManagementService>().createEntity(1, entitySchema, entity, users)
            lastUUID = outEntity.id
        }
        // Act
        val filters = listOf(
            AttributeFilter(
                schema = AttributeSchema(
                    type = AttributeTypeImpl.INT,
                    fieldNumber = -1,
                    name = "internal_id",
                    isIndexed = false
                ),
                filter = NumberRangeFilter(minValue = 1.0, includeMin = false, maxValue = 4.0, includeMax = true)
            )
        )
        val entities = entityManagementService.listEntities(1, entitySchema, filters)

        // Assert
        assertEquals(3, entities.size)
        assertTrue(entities.any { it.attributeValues[0] == "Author4" })
        assertFalse(entities.any { it.attributeValues[0] == "Author1" })
    }

    @Test
    fun `list all entities with not equality filter on id throws`() {
        // Arrange
        val entityManagementService = get<EntityManagementService>()
        var lastUUID = UUID.randomUUID()
        (1..5).forEach {
            val dt = Instant.now().truncatedTo(ChronoUnit.MILLIS)
            val entity = createEntity(entitySchema,dt,"Author$it", it, false, UUID.randomUUID(), it.toFloat(), it.toDouble())
            val users = listOf(EntityUser(userId = 1, role = EntityUserRoleEnum.AUTHOR))
            val outEntity = get<EntityManagementService>().createEntity(1, entitySchema, entity, users)
            lastUUID = outEntity.id
        }
        // Act
        val filters = listOf(
            AttributeFilter(
                schema = AttributeSchema(
                    type = AttributeTypeImpl.STRING,
                    fieldNumber = 123,
                    name = "id",
                    isIndexed = false
                ),
                filter = NumberRangeFilter(minValue = 0.0)
            )
        )
        assertThrows<IllegalStateException> { entityManagementService.listEntities(1, entitySchema, filters) }
    }

    @Test
    fun `list all entities with range filter min excluding on int dynamic attribute`() {
        // Arrange
        val entityManagementService = get<EntityManagementService>()
        (1..20).forEach {
            val dt = Instant.now().truncatedTo(ChronoUnit.MILLIS)
            val entity = createEntity(entitySchema,dt,"Author$it", it, false, UUID.randomUUID(), it.toFloat(), it.toDouble())
            val users = listOf(EntityUser(userId = 1, role = EntityUserRoleEnum.AUTHOR))
            get<EntityManagementService>().createEntity(1, entitySchema, entity, users)
        }
        // Act
        val filters = listOf(
            AttributeFilter(
                schema = AttributeSchema(
                    type = AttributeTypeImpl.INT,
                    fieldNumber = 1,
                    name = "Severity",
                    isIndexed = true
                ),
                filter = NumberRangeFilter(minValue = 5.0, includeMin = false)
            )
        )
        val entities = entityManagementService.listEntities(1, entitySchema, filters)

        // Assert
        assertEquals(15, entities.size)
    }

    @Test
    fun `list all entities with range filter min including on int dynamic attribute`() {
        // Arrange
        val entityManagementService = get<EntityManagementService>()
        (1..20).forEach {
            val dt = Instant.now().truncatedTo(ChronoUnit.MILLIS)
            val entity = createEntity(entitySchema,dt,"Author$it", it, false, UUID.randomUUID(), it.toFloat(), it.toDouble())
            val users = listOf(EntityUser(userId = 1, role = EntityUserRoleEnum.AUTHOR))
            get<EntityManagementService>().createEntity(1, entitySchema, entity, users)
        }
        // Act
        val filters = listOf(
            AttributeFilter(
                schema = AttributeSchema(
                    type = AttributeTypeImpl.INT,
                    fieldNumber = 1,
                    name = "Severity",
                    isIndexed = true
                ),
                filter = NumberRangeFilter(minValue = 5.0, includeMin = true)
            )
        )
        val entities = entityManagementService.listEntities(1, entitySchema, filters)

        // Assert
        assertEquals(16, entities.size)
    }

    @Test
    fun `list all entities with range filter max excluding on int dynamic attribute`() {
        // Arrange
        val entityManagementService = get<EntityManagementService>()
        (1..20).forEach {
            val dt = Instant.now().truncatedTo(ChronoUnit.MILLIS)
            val entity = createEntity(entitySchema,dt,"Author$it", it, false, UUID.randomUUID(), it.toFloat(), it.toDouble())
            val users = listOf(EntityUser(userId = 1, role = EntityUserRoleEnum.AUTHOR))
            get<EntityManagementService>().createEntity(1, entitySchema, entity, users)
        }
        // Act
        val filters = listOf(
            AttributeFilter(
                schema = AttributeSchema(
                    type = AttributeTypeImpl.INT,
                    fieldNumber = 1,
                    name = "Severity",
                    isIndexed = true
                ),
                filter = NumberRangeFilter(maxValue = 5.0, includeMax = false)
            )
        )
        val entities = entityManagementService.listEntities(1, entitySchema, filters)

        // Assert
        assertEquals(4, entities.size)
    }

    @Test
    fun `list all entities with range filter max including on int dynamic attribute`() {
        // Arrange
        val entityManagementService = get<EntityManagementService>()
        (1..20).forEach {
            val dt = Instant.now().truncatedTo(ChronoUnit.MILLIS)
            val entity = createEntity(entitySchema,dt,"Author$it", it, false, UUID.randomUUID(), it.toFloat(), it.toDouble())
            val users = listOf(EntityUser(userId = 1, role = EntityUserRoleEnum.AUTHOR))
            get<EntityManagementService>().createEntity(1, entitySchema, entity, users)
        }
        // Act
        val filters = listOf(
            AttributeFilter(
                schema = AttributeSchema(
                    type = AttributeTypeImpl.INT,
                    fieldNumber = 1,
                    name = "Severity",
                    isIndexed = true
                ),
                filter = NumberRangeFilter(maxValue = 5.0, includeMax = true)
            )
        )
        val entities = entityManagementService.listEntities(1, entitySchema, filters)

        // Assert
        assertEquals(5, entities.size)
    }

    @Test
    fun `list all entities with range filter max including and min excluding on int dynamic attribute`() {
        // Arrange
        val entityManagementService = get<EntityManagementService>()
        (1..20).forEach {
            val dt = Instant.now().truncatedTo(ChronoUnit.MILLIS)
            val entity = createEntity(entitySchema,dt,"Author$it", it, false, UUID.randomUUID(), it.toFloat(), it.toDouble())
            val users = listOf(EntityUser(userId = 1, role = EntityUserRoleEnum.AUTHOR))
            get<EntityManagementService>().createEntity(1, entitySchema, entity, users)
        }
        // Act
        val filters = listOf(
            AttributeFilter(
                schema = AttributeSchema(
                    type = AttributeTypeImpl.INT,
                    fieldNumber = 1,
                    name = "Severity",
                    isIndexed = true
                ),
                filter = NumberRangeFilter(minValue = 3.0, includeMin = false, maxValue = 5.0, includeMax = true)
            )
        )
        val entities = entityManagementService.listEntities(1, entitySchema, filters)

        // Assert
        assertEquals(2, entities.size)
    }

    @Test
    fun `list all entities with empty range on int dynamic attribute`() {
        // Arrange
        val entityManagementService = get<EntityManagementService>()
        (1..20).forEach {
            val dt = Instant.now().truncatedTo(ChronoUnit.MILLIS)
            val entity = createEntity(entitySchema,dt,"Author$it", it, false, UUID.randomUUID(), it.toFloat(), it.toDouble())
            val users = listOf(EntityUser(userId = 1, role = EntityUserRoleEnum.AUTHOR))
            get<EntityManagementService>().createEntity(1, entitySchema, entity, users)
        }
        // Act
        val filters = listOf(
            AttributeFilter(
                schema = AttributeSchema(
                    type = AttributeTypeImpl.INT,
                    fieldNumber = 1,
                    name = "Severity",
                    isIndexed = true
                ),
                filter = NumberRangeFilter(minValue = 6.0, includeMin = false, maxValue = 5.0, includeMax = true)
            )
        )
        val entities = entityManagementService.listEntities(1, entitySchema, filters)

        // Assert
        assertEquals(0, entities.size)
    }

    @Test
    fun `list all entities with range filter max including and min excluding on float dynamic attribute`() {
        // Arrange
        val entityManagementService = get<EntityManagementService>()
        (1..20).forEach {
            val dt = Instant.now().truncatedTo(ChronoUnit.MILLIS)
            val entity = createEntity(entitySchema,dt,"Author$it", it, false, UUID.randomUUID(), it.toFloat(), it.toDouble())
            val users = listOf(EntityUser(userId = 1, role = EntityUserRoleEnum.AUTHOR))
            get<EntityManagementService>().createEntity(1, entitySchema, entity, users)
        }
        // Act
        val filters = listOf(
            AttributeFilter(
                schema = AttributeSchema(
                    name = "Float",
                    type = AttributeTypeImpl.FLOAT,
                    fieldNumber = 5,
                    isIndexed = true
                ),
                filter = NumberRangeFilter(minValue = 3.0, includeMin = false, maxValue = 5.0, includeMax = true)
            )
        )
        val entities = entityManagementService.listEntities(1, entitySchema, filters)

        // Assert
        assertEquals(2, entities.size)
    }

    @Test
    fun `list entities with list filter on string dynamic attribute`() {
        // Arrange
        val entityManagementService = get<EntityManagementService>()
        (1..20).forEach {
            val dt = Instant.now().truncatedTo(ChronoUnit.MILLIS)
            val entity = createEntity(entitySchema,dt,"Author$it", it, false, UUID.randomUUID(), it.toFloat(), it.toDouble())
            val users = listOf(EntityUser(userId = 1, role = EntityUserRoleEnum.AUTHOR))
            get<EntityManagementService>().createEntity(1, entitySchema, entity, users)
        }
        // Act
        val filters = listOf(
            AttributeFilter(
                schema = AttributeSchema(
                    type = AttributeTypeImpl.STRING,
                    fieldNumber = 0,
                    name = "Author",
                    isIndexed = true
                ),
                filter = ValueListFilter(valueList = listOf("Author2", "Author11", "Author18"))
            )
        )
        val entities = entityManagementService.listEntities(1, entitySchema, filters)

        // Assert
        assertEquals(3, entities.size)
        assertTrue(entities.any { it.attributeValues[1] == 2 } && entities.any { it.attributeValues[1] == 11 } && entities.any { it.attributeValues[1] == 18 })
    }

    @Test
    fun `list entities with list filter on int dynamic attribute`() {
        // Arrange
        val entityManagementService = get<EntityManagementService>()
        (1..20).forEach {
            val dt = Instant.now().truncatedTo(ChronoUnit.MILLIS)
            val entity = createEntity(entitySchema,dt,"Author$it", it, false, UUID.randomUUID(), it.toFloat(), it.toDouble())
            val users = listOf(EntityUser(userId = 1, role = EntityUserRoleEnum.AUTHOR))
            get<EntityManagementService>().createEntity(1, entitySchema, entity, users)
        }
        // Act
        val filters = listOf(
            AttributeFilter(
                schema = AttributeSchema(
                    type = AttributeTypeImpl.INT,
                    fieldNumber = 1,
                    name = "Severity",
                    isIndexed = true
                ),
                filter = ValueListFilter(valueList = listOf(3, 5, 7))
            )
        )
        val entities = entityManagementService.listEntities(1, entitySchema, filters)

        // Assert
        assertEquals(3, entities.size)
        assertTrue(entities.any { it.attributeValues[1] == 3 } && entities.any { it.attributeValues[1] == 5 } && entities.any { it.attributeValues[1] == 7 })
    }

    @Test
    fun `list all entities with list filter on id`() {
        // Arrange
        val entityManagementService = get<EntityManagementService>()
        var uuids = mutableListOf<UUID>()
        (1..10).forEach {
            val dt = Instant.now().truncatedTo(ChronoUnit.MILLIS)
            val entity = createEntity(entitySchema,dt,"Author$it", it, false, UUID.randomUUID(), it.toFloat(), it.toDouble())
            val users = listOf(EntityUser(userId = 1, role = EntityUserRoleEnum.AUTHOR))
            val outEntity = get<EntityManagementService>().createEntity(1, entitySchema, entity, users)
            uuids.add(outEntity.id!!)
        }
        // Act
        val filters = listOf(
            AttributeFilter(
                schema = AttributeSchema(
                    type = AttributeTypeImpl.STRING,
                    fieldNumber = 123,
                    name = "id",
                    isIndexed = false
                ),
                filter = ValueListFilter(valueList = uuids.filterIndexed { index, _ -> index % 3 == 0 })
            )
        )
        val entities = entityManagementService.listEntities(1, entitySchema, filters)

        // Assert
        assertEquals(4, entities.size)
        assertTrue(
            entities.any { it.attributeValues[1] == 1 } &&
            entities.any { it.attributeValues[1] == 4 } &&
            entities.any { it.attributeValues[1] == 7 } &&
            entities.any { it.attributeValues[1] == 10 })
    }

    @Test
    fun `list all entities with list filter on internal_id`() {
        // Arrange
        val entityManagementService = get<EntityManagementService>()
        (1..10).forEach {
            val dt = Instant.now().truncatedTo(ChronoUnit.MILLIS)
            val entity = createEntity(entitySchema,dt,"Author$it", it, false, UUID.randomUUID(), it.toFloat(), it.toDouble())
            val users = listOf(EntityUser(userId = 1, role = EntityUserRoleEnum.AUTHOR))
            val outEntity = get<EntityManagementService>().createEntity(1, entitySchema, entity, users)
        }
        // Act
        val filters = listOf(
            AttributeFilter(
                schema = AttributeSchema(
                    type = AttributeTypeImpl.INT,
                    fieldNumber = -1,
                    name = "internal_id",
                    isIndexed = false
                ),
                filter = ValueListFilter(valueList = listOf(2,6,8))
            )
        )
        val entities = entityManagementService.listEntities(1, entitySchema, filters)

        // Assert
        assertEquals(3, entities.size)
        assertTrue(
            entities.any { it.attributeValues[1] == 2 } &&
                entities.any { it.attributeValues[1] == 6 } &&
                entities.any { it.attributeValues[1] == 8 })
    }
}

