package ru.yandex.crm.apphost.kotlin.handlers.entitystorage.ut.service

import org.junit.jupiter.api.Assertions.assertEquals
import org.junit.jupiter.api.BeforeEach
import org.junit.jupiter.api.Test
import org.junit.jupiter.api.assertThrows
import org.koin.test.get
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.IndexRepository
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.getEntityRepository
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.Entity
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.EntitySchema
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.EntityUser
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.EntityUserRoleEnum
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.service.EntityManagementService
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.ut.EntityDbBackedTests
import ru.yandex.crm.library.kotlin.database.hibernate.getRepository
import ru.yandex.crm.library.kotlin.database.hibernate.transaction
import java.time.Instant
import java.time.temporal.ChronoUnit
import java.util.UUID
import kotlin.test.assertNotNull

open class EntityManagementServiceTests : EntityDbBackedTests() {

    lateinit var entitySchema: EntitySchema

    @BeforeEach
    fun beforeEach() {
        entitySchema = createEntitySchema()
    }

    @Test
    fun `create entity`() {
        // Arrange
        val instant = Instant.now().truncatedTo(ChronoUnit.MILLIS)
        val uuid = UUID.fromString("973545e4-f09d-11ec-8ea0-0242ac120002")
        val departmentId = UUID.fromString("779b8307-b6d8-4aff-8fd0-7e3dc260e570")
        val entity = createEntity(entitySchema,instant,"John Doe", 99, false, uuid, departmentId = departmentId)
        val users = listOf(EntityUser(userId = 1, role = EntityUserRoleEnum.AUTHOR))

        // Act
        val outEntity = get<EntityManagementService>().createEntity(1, entitySchema, entity, users)

        // Assert
        assertNotNull(outEntity)
        assertNotNull(outEntity.id)

        transaction {
            val repository = getEntityRepository(entitySchema)
            val entity = repository.findOne(outEntity.id!!)
            assertNotNull(entity)
            assertEquals(1, entity.internalId)
            assertEquals(departmentId, entity.departmentId)
            assertEquals("John Doe", entity.attributeValues[0])
            assertEquals(99, entity.attributeValues[1])
            assertEquals(instant, entity.attributeValues[2])
            assertEquals(false, entity.attributeValues[3])
            assertEquals(uuid, entity.attributeValues[4])
        }
    }

    @Test
    fun `create entity without author causes error`(){
        //Arrange
        val instant = Instant.now().truncatedTo(ChronoUnit.MILLIS)
        val uuid = UUID.fromString("973545e4-f09d-11ec-8ea0-0242ac120002")
        val departmentId = UUID.fromString("779b8307-b6d8-4aff-8fd0-7e3dc260e570")
        val entity = createEntity(entitySchema,instant,"John Doe", 99, false, uuid, departmentId = departmentId)

        //Act
        assertThrows<IllegalStateException> { get<EntityManagementService>().createEntity(1, entitySchema, entity, listOf()) }

    }

    @Test
    fun `create entity with multiple authors causes error`(){
        //Arrange
        val instant = Instant.now().truncatedTo(ChronoUnit.MILLIS)
        val uuid = UUID.fromString("973545e4-f09d-11ec-8ea0-0242ac120002")
        val departmentId = UUID.fromString("779b8307-b6d8-4aff-8fd0-7e3dc260e570")
        val entity = createEntity(entitySchema,instant,"John Doe", 99, false, uuid, departmentId = departmentId)

        //Act
        assertThrows<IllegalStateException> { get<EntityManagementService>().createEntity(1, entitySchema, entity,
            listOf(
                EntityUser(userId = 1, role = EntityUserRoleEnum.AUTHOR),
                EntityUser(userId = 2, role = EntityUserRoleEnum.AUTHOR))) }
    }


    @Test
    fun `create entity with index cause writing to index table`() {
        //Arrange
        val instant = Instant.now().truncatedTo(ChronoUnit.MILLIS)
        val uuid = UUID.fromString("973545e4-f09d-11ec-8ea0-0242ac120002")
        val entity = createEntity(entitySchema,instant,"John Doe", 99, false, uuid,3.1415f, 2.7182818284590)
        val users = listOf(EntityUser(userId = 1, role = EntityUserRoleEnum.AUTHOR))
        // Act
        get<EntityManagementService>().createEntity(1, entitySchema,  entity, users)

        //Assert

        transaction {
            val indexRepository = getRepository<IndexRepository>()
            val indexes = indexRepository.findAll()
            assertEquals(7, indexes.size)
            assertEquals("John Doe", indexes.first { it.fieldNumber == 0 }.stringValue)
            assertEquals(99, indexes.first { it.fieldNumber == 1 }.intValue)
            assertEquals(instant, indexes.first { it.fieldNumber == 2 }.timestampValue)
            assertEquals(false, indexes.first { it.fieldNumber == 3 }.boolValue)
            assertEquals(uuid, indexes.first { it.fieldNumber == 4 }.uuidValue)
            assertEquals(3.1415f, indexes.first { it.fieldNumber == 5 }.floatValue)
            assertEquals(2.7182818284590, indexes.first { it.fieldNumber == 6 }.doubleValue)
        }
    }

    @Test
    fun `update entity`() {
        // Arrange
        val dt = Instant.now().truncatedTo(ChronoUnit.MILLIS)
        val id = transaction {
            val repository = getEntityRepository(entitySchema)
            var entityObject = createEntity(entitySchema, dt)
            entityObject = repository.create(entityObject)
            entityObject.id
        }
        // Act
        val updatedAttributes = mapOf(0 to "Tolstoy")

        val entityManagementService = get<EntityManagementService>()
        val outEntity = entityManagementService.updateEntity(1, entitySchema, id!!, 1, updatedAttributes)

        transaction {
            val dbEntity = getEntityRepository(entitySchema).findOne(id)

            // Assert
            assertEquals("Tolstoy", outEntity.attributeValues[0])
            assertEquals(50, outEntity.attributeValues[1])
            assertEquals(dt, outEntity.attributeValues[2])

            assertEquals("Tolstoy", dbEntity!!.attributeValues[0])
            assertEquals(50, dbEntity.attributeValues[1])
            assertEquals(dt, dbEntity.attributeValues[2])
        }
    }

    @Test
    fun `delete entity`() {
        // Arrange
        val dt = Instant.now().truncatedTo(ChronoUnit.MILLIS)
        val id = transaction {
            val repository = getEntityRepository(entitySchema)
            var entityObject = createEntity(entitySchema, dt)
            entityObject = repository.create(entityObject)
            entityObject.id
        }

        // Act
        val entityManagementService = get<EntityManagementService>()
        entityManagementService.deleteEntity(1, entitySchema, id!!)

        // Assert
        val dbEntity = transaction {
            getEntityRepository(entitySchema).findOne(id)
        }
        assertEquals(true, dbEntity?.isDeleted)
    }

    @Test
    fun `list all entities`() {
        // Arrange
        transaction {
            val repository = getEntityRepository(entitySchema)
            (1..5).forEach {
                val dt = Instant.now().truncatedTo(ChronoUnit.MILLIS)
                val entityObject = createEntity(entitySchema, dt, "Author$it", it, false, UUID.randomUUID(), it.toFloat(), it.toDouble())
                repository.create(entityObject)
            }
        }
        // Act
        val entityManagementService = get<EntityManagementService>()
        val entities = entityManagementService.listEntities(1, entitySchema, emptyList())

        // Assert
        assertEquals(5, entities.size)
        assertEquals(7, entities[0].attributeValues.size)
    }
}
