package ru.yandex.crm.apphost.kotlin.handlers.entitystorage.ut.service

import org.junit.jupiter.api.BeforeEach
import org.junit.jupiter.api.Test
import org.junit.jupiter.api.assertThrows
import org.koin.test.get
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.EntityUserRepository
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.EntitySchema
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.EntityUser
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.repository.model.EntityUserRoleEnum
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.service.EntityManagementService
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.service.EntityUserManager
import ru.yandex.crm.apphost.kotlin.handlers.entitystorage.ut.EntityDbBackedTests
import ru.yandex.crm.library.kotlin.database.hibernate.getRepository
import ru.yandex.crm.library.kotlin.database.hibernate.transaction
import java.time.Instant
import java.time.temporal.ChronoUnit
import java.util.UUID
import kotlin.test.assertEquals

class EntityUserServiceTests : EntityDbBackedTests() {

    lateinit var entitySchema: EntitySchema

    @BeforeEach
    fun beforeEach() {
        entitySchema = createEntitySchema()
    }


    @Test
    fun `add author user to entity`() {
        // Arrange
        val instant = Instant.now().truncatedTo(ChronoUnit.MILLIS)
        val uuid = UUID.fromString("973545e4-f09d-11ec-8ea0-0242ac120002")
        val departmentId = UUID.fromString("779b8307-b6d8-4aff-8fd0-7e3dc260e570")
        val entity = createEntity(entitySchema, instant,"John Doe", 99, false, uuid, departmentId = departmentId)
        val users = listOf(EntityUser(userId = 1, role = EntityUserRoleEnum.AUTHOR))
        val outEntity = get<EntityManagementService>().createEntity(1, entitySchema, entity, users)

        // Act
        get<EntityUserManager>().addEntityUser(EntityUser(entityId = outEntity.id!!, userId = 2, role = EntityUserRoleEnum.AUTHOR))

        //Assert
        transaction {
            val entityUserRepository = getRepository<EntityUserRepository>()
            val responsibleUsers = entityUserRepository.getEntityUsers(outEntity.id!!, EntityUserRoleEnum.AUTHOR)
            assertEquals(2, responsibleUsers.size)
        }
    }

    @Test
    fun `unable to add existed user to entity`() {
        // Arrange
        val instant = Instant.now().truncatedTo(ChronoUnit.MILLIS)
        val uuid = UUID.fromString("973545e4-f09d-11ec-8ea0-0242ac120002")
        val departmentId = UUID.fromString("779b8307-b6d8-4aff-8fd0-7e3dc260e570")
        val entity = createEntity(entitySchema, instant,"John Doe", 99, false, uuid, departmentId = departmentId)
        val users = listOf(EntityUser(userId = 1, role = EntityUserRoleEnum.AUTHOR))
        val outEntity = get<EntityManagementService>().createEntity(1, entitySchema, entity, users)

        // Act
        assertThrows<IllegalStateException> { get<EntityUserManager>().addEntityUser(
            EntityUser(entityId = outEntity.id!!, userId = 1, role = EntityUserRoleEnum.AUTHOR)) }
    }

    @Test
    fun `unable to remove one author user from entity`() {
        // Arrange
        val instant = Instant.now().truncatedTo(ChronoUnit.MILLIS)
        val uuid = UUID.fromString("973545e4-f09d-11ec-8ea0-0242ac120002")
        val departmentId = UUID.fromString("779b8307-b6d8-4aff-8fd0-7e3dc260e570")
        val entity = createEntity(entitySchema, instant,"John Doe", 99, false, uuid, departmentId = departmentId)
        val users = listOf(EntityUser(userId = 1, role = EntityUserRoleEnum.AUTHOR))
        val outEntity = get<EntityManagementService>().createEntity(1, entitySchema, entity, users)

        // Act
        assertThrows<IllegalStateException> { get<EntityUserManager>().removeEntityUser(
            EntityUser(entityId = outEntity.id!!, userId = 1, role = EntityUserRoleEnum.AUTHOR)) }
    }

    @Test
    fun `add responsible user to entity`() {
        // Arrange
        val instant = Instant.now().truncatedTo(ChronoUnit.MILLIS)
        val uuid = UUID.fromString("973545e4-f09d-11ec-8ea0-0242ac120002")
        val departmentId = UUID.fromString("779b8307-b6d8-4aff-8fd0-7e3dc260e570")
        val entity = createEntity(entitySchema, instant,"John Doe", 99, false, uuid, departmentId = departmentId)
        val users = listOf(EntityUser(userId = 1, role = EntityUserRoleEnum.AUTHOR))
        val outEntity = get<EntityManagementService>().createEntity(1, entitySchema, entity, users)

        // Act
        get<EntityUserManager>().addEntityUser(EntityUser(entityId = outEntity.id!!, userId = 2, role = EntityUserRoleEnum.RESPONSIBLE))

        //Assert
        transaction {
            val entityUserRepository = getRepository<EntityUserRepository>()
            val responsibleUsers = entityUserRepository.getEntityUsers(outEntity.id!!, EntityUserRoleEnum.RESPONSIBLE)
            assertEquals(1, responsibleUsers.size)
            assertEquals(outEntity.id, responsibleUsers[0].entityId)
            assertEquals(2, responsibleUsers[0].userId)
        }
    }

    @Test
    fun `add another responsible user of entity`() {
        // Arrange
        val instant = Instant.now().truncatedTo(ChronoUnit.MILLIS)
        val uuid = UUID.fromString("973545e4-f09d-11ec-8ea0-0242ac120002")
        val departmentId = UUID.fromString("779b8307-b6d8-4aff-8fd0-7e3dc260e570")
        val entity = createEntity(entitySchema, instant,"John Doe", 99, false, uuid, departmentId = departmentId)
        val users = listOf(EntityUser(userId = 1, role = EntityUserRoleEnum.AUTHOR))
        val outEntity = get<EntityManagementService>().createEntity(1, entitySchema, entity, users)
        get<EntityUserManager>().addEntityUser(EntityUser(entityId = outEntity.id!!, userId = 2, role = EntityUserRoleEnum.RESPONSIBLE))

        // Act
        get<EntityUserManager>().addEntityUser(EntityUser(entityId = outEntity.id!!, userId = 3, role = EntityUserRoleEnum.RESPONSIBLE))

        //Assert
        transaction {
            val entityUserRepository = getRepository<EntityUserRepository>()
            val responsibleUsers = entityUserRepository.getEntityUsers(outEntity.id!!, EntityUserRoleEnum.RESPONSIBLE)
            assertEquals(2, responsibleUsers.size)
        }
    }

    @Test
    fun `remove responsible user from entity`() {
        // Arrange
        val instant = Instant.now().truncatedTo(ChronoUnit.MILLIS)
        val uuid = UUID.fromString("973545e4-f09d-11ec-8ea0-0242ac120002")
        val departmentId = UUID.fromString("779b8307-b6d8-4aff-8fd0-7e3dc260e570")
        val entity = createEntity(entitySchema, instant,"John Doe", 99, false, uuid, departmentId = departmentId)
        val users = listOf(EntityUser(userId = 1, role = EntityUserRoleEnum.AUTHOR))
        val outEntity = get<EntityManagementService>().createEntity(1, entitySchema, entity, users)
        get<EntityUserManager>().addEntityUser(EntityUser(entityId = outEntity.id!!, userId = 2, role = EntityUserRoleEnum.RESPONSIBLE))

        // Act
        get<EntityUserManager>().removeEntityUser(EntityUser(entityId = outEntity.id!!, userId = 2, role = EntityUserRoleEnum.RESPONSIBLE))

        //Assert
        transaction {
            val entityUserRepository = getRepository<EntityUserRepository>()
            val responsibleUsers = entityUserRepository.getEntityUsers(outEntity.id!!, EntityUserRoleEnum.RESPONSIBLE)
            assertEquals(0, responsibleUsers.size)
        }
    }


    @Test
    fun `unable to remove non-existed user from entity`() {
        // Arrange
        val instant = Instant.now().truncatedTo(ChronoUnit.MILLIS)
        val uuid = UUID.fromString("973545e4-f09d-11ec-8ea0-0242ac120002")
        val departmentId = UUID.fromString("779b8307-b6d8-4aff-8fd0-7e3dc260e570")
        val entity = createEntity(entitySchema, instant,"John Doe", 99, false, uuid, departmentId = departmentId)
        val users = listOf(EntityUser(userId = 1, role = EntityUserRoleEnum.AUTHOR))
        val outEntity = get<EntityManagementService>().createEntity(1, entitySchema, entity, users)
        get<EntityUserManager>().addEntityUser(EntityUser(entityId = outEntity.id!!, userId = 2, role = EntityUserRoleEnum.RESPONSIBLE))

        // Act
        assertThrows<IllegalStateException> {  get<EntityUserManager>().removeEntityUser(
            EntityUser(entityId = outEntity.id!!, userId = 4, role = EntityUserRoleEnum.OBSERVER))}

    }
}
