package ru.yandex.crm.apphost.kotlin.handlers.organizationmanager.ut.service

import com.google.protobuf.Int64Value
import org.apache.commons.dbcp2.BasicDataSource
import org.junit.jupiter.api.AfterEach
import org.junit.jupiter.api.BeforeEach
import org.junit.jupiter.api.Test
import org.junit.jupiter.api.assertThrows
import org.koin.core.context.startKoin
import org.koin.core.context.stopKoin
import org.koin.dsl.bind
import org.koin.dsl.module
import org.koin.test.KoinTest
import org.koin.test.inject
import ru.yandex.crm.apphost.kotlin.dal.organizationmanager.schema.OrganizationManagerSchema
import ru.yandex.crm.apphost.kotlin.dal.usermanager.schema.UserManagerSchema
import ru.yandex.crm.apphost.kotlin.handlers.organizationmanager.repository.impl.OrganizationRepositoryImpl
import ru.yandex.crm.apphost.kotlin.handlers.organizationmanager.service.OrganizationService
import ru.yandex.crm.apphost.kotlin.handlers.organizationmanager.service.impl.OrganizationServiceImpl
import ru.yandex.crm.apphost.kotlin.handlers.organizationmanager.service.mappers.OrganizationEntityMapper
import ru.yandex.crm.apphost.kotlin.handlers.organizationmanager.service.mappers.impl.OrganizationEntityMapperImpl
import ru.yandex.crm.apphost.kotlin.handlers.organizationmanager.ut.repository.MockOrganizationRepository
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateConfig
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateSchema
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateSessionManager
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateSessionManagerImpl
import ru.yandex.crm.library.kotlin.test.isNotNull
import ru.yandex.crm.proto.gallifrey.organizationmanager.Organizationmanager.*
import javax.sql.DataSource
import kotlin.test.assertNotNull
import kotlin.test.assertNull

private val hibernateConfig = HibernateConfig().apply {
    dialect = "org.hibernate.dialect.H2Dialect"
    showSql = true
    hbm2ddlAuto = "create"
}

private val dataSource = BasicDataSource().apply {
    driverClassName = "org.h2.Driver"
    url = "jdbc:h2:mem:myDb;DB_CLOSE_DELAY=-1"
}

private val module = module {
    single { hibernateConfig } bind HibernateConfig::class
    single { dataSource } bind DataSource::class
    single { HibernateSessionManagerImpl(get(), get(), getAll()) } bind HibernateSessionManager::class
    single { OrganizationManagerSchema() } bind HibernateSchema::class
    single { UserManagerSchema() } bind HibernateSchema::class
    factory { params -> MockOrganizationRepository(params.get()) } bind OrganizationRepositoryImpl::class
    single<OrganizationEntityMapper> { OrganizationEntityMapperImpl() }

    factory { OrganizationServiceImpl(get()) } bind OrganizationService::class
}

class OrganizationServiceTests : KoinTest {

    private val service: OrganizationService by inject()

    @BeforeEach
    fun beforeEach() {
        startKoin {
            modules(module)
        }
    }

    @AfterEach
    fun afterEach() {
        stopKoin()
    }

    @Test
    fun `get all organizations`() {
        val organizations = service.getAllOrganizations()

        assert(organizations.size == 5)
        assert(organizations.all { it.id.isNotNull() })
        assert(organizations.all { it.data.pool.isNotNull() })
        assert(organizations.all { it.data.slug.isNotNull() })
        assert(organizations.all { it.data.name.isNotNull() })
    }

    @Test
    fun `get existing organization by id`() {
        val organization1 = service.getOrganization(1)
        val organization2 = service.getOrganization(2)
        val organization3 = service.getOrganization(3)

        assert(organization1.id.value == 1L)
        assert(organization2.id.value == 2L)
        assert(organization3.id.value == 3L)

        assert(organization1.data.pool.isNotNull())
        assert(organization1.data.slug.isNotNull())
        assert(organization1.data.name.isNotNull())

        assert(organization2.data.pool.isNotNull())
        assert(organization2.data.slug.isNotNull())
        assert(organization2.data.name.isNotNull())

        assert(organization3.data.pool.isNotNull())
        assert(organization3.data.slug.isNotNull())
        assert(organization3.data.name.isNotNull())
    }

    @Test
    fun `get not existing organization by id`() {
        assertThrows<Exception> {
            service.getOrganization(-1)
        }
    }

    @Test
    fun `get existing organization by id or null`() {
        val organization1 = service.getOrganizationOrNull(1)
        val organization2 = service.getOrganizationOrNull(-1)

        assertNotNull(organization1)
        assertNull(organization2)

        assert(organization1.id.value == 1L)
        assert(organization1.data.pool.isNotNull())
        assert(organization1.data.slug.isNotNull())
        assert(organization1.data.name.isNotNull())
    }

    @Test
    fun `create organization`() {
        val data = OrganizationData.newBuilder()
            .setName("name")
            .setSlug("slug")
            .setPool(1)
            .build()

        val savedOrganization = service.createOrganization(data)

        assertNotNull(savedOrganization)
        assertNotNull(savedOrganization.id)
        assertNotNull(savedOrganization.data.name)
        assertNotNull(savedOrganization.data.slug)
        assertNotNull(savedOrganization.data.pool)
    }

    @Test
    fun `update organization`() {
        val organization = UpdateOrganizationData.newBuilder()
            .setNewName("new name")
            .setNewSlug("new slug")
            .setNewPool(10L)
            .build()

        val oldOrganization = service.getOrganization(1)
        val updatedOrganization = service.updateOrganization(1, organization)

        assert(oldOrganization.id.value == 1L)
        assert(oldOrganization.data.name == "name")
        assert(oldOrganization.data.slug == "slug1")
        assert(oldOrganization.data.pool == 1L)

        assertNotNull(updatedOrganization)
        assert(updatedOrganization.id.value == 1L)
        assert(updatedOrganization.data.name == "new name")
        assert(updatedOrganization.data.slug == "new slug")
        assert(updatedOrganization.data.pool == 10L)
    }
}
