package ru.yandex.crm.apphost.kotlin.handlers.teammanager.service.impl

import ru.yandex.crm.apphost.kotlin.handlers.teammanager.repository.OrganizationTeamRepository
import ru.yandex.crm.apphost.kotlin.handlers.teammanager.service.OrganizationTeamService
import ru.yandex.crm.apphost.kotlin.handlers.teammanager.service.mappers.OrganizationTeamMapper
import ru.yandex.crm.library.kotlin.database.hibernate.getRepository
import ru.yandex.crm.library.kotlin.database.hibernate.transaction
import ru.yandex.crm.proto.gallifrey.teammanager.Teammanager.Team
import ru.yandex.crm.proto.gallifrey.teammanager.Teammanager.TeamData
import ru.yandex.crm.proto.gallifrey.teammanager.Teammanager.UpdateTeamData
import java.util.UUID

class OrganizationTeamServiceImpl(
    private val mapper: OrganizationTeamMapper
): OrganizationTeamService {

    override fun getAllOrganizationTeams(organizationId: Long): List<Team> {
        val teams = transaction {
            val repository = getRepository<OrganizationTeamRepository>()
            repository.getAllOrganizationTeams(organizationId)
        }

        return teams.map { mapper.toProtobufModel(it) }
    }

    override fun getOrganizationTeamById(organizationId: Long, teamId: UUID): Team {
        val team = transaction {
            val repository = getRepository<OrganizationTeamRepository>()
            repository.getOrganizationTeamById(organizationId, teamId)
        } ?: error("The Organization teams with $teamId was not found")

        return mapper.toProtobufModel(team)
    }

    override fun createOrganizationTeam(team: TeamData): Team {
        val newTeam = transaction {
            val repository = getRepository<OrganizationTeamRepository>()
            val newTeam = mapper.fromProtobufModel(team)
            repository.save(newTeam)
        }

        return mapper.toProtobufModel(newTeam)
    }

    override fun updateOrganizationTeam(
        updateTeamData: UpdateTeamData,
        organizationId: Long,
        teamId: UUID
    ): Team {
        val team = transaction {
            val repository = getRepository<OrganizationTeamRepository>()
            val targetTeam = repository.getOrganizationTeamById(organizationId, teamId)
            if (targetTeam === null) {
                error("The Organization team with $teamId was not found")
            }

            val newNames = updateTeamData.newNamesList.map {
                mapper.fromProtobufModel(it, targetTeam)
            }.toMutableList()

            newNames.forEach {
                val existingName = targetTeam.names.firstOrNull { name -> it.id!! == name.id }
                if (existingName != null) {
                    existingName.name = it.name
                } else {
                    targetTeam.names.add(it)
                }
            }

            repository.save(targetTeam)
        }

        return mapper.toProtobufModel(team)
    }

    override fun deleteOrganizationTeam(organizationId: Long, teamId: UUID) {
        transaction {
            val repository = getRepository<OrganizationTeamRepository>()
            val team = repository.getOrganizationTeamById(organizationId, teamId)
            if (team !== null) {
                team.isDeleted = true
                repository.save(team)
            }
        }
    }
}
