package ru.yandex.crm.apphost.kotlin.handlers.teammanager.ut.service

import org.apache.commons.dbcp2.BasicDataSource
import org.junit.jupiter.api.AfterEach
import org.junit.jupiter.api.BeforeEach
import org.junit.jupiter.api.Test
import org.junit.jupiter.api.assertThrows
import org.koin.core.context.startKoin
import org.koin.core.context.stopKoin
import org.koin.dsl.bind
import org.koin.dsl.module
import org.koin.test.KoinTest
import org.koin.test.inject
import ru.yandex.crm.apphost.kotlin.dal.organizationmanager.schema.OrganizationManagerSchema
import ru.yandex.crm.apphost.kotlin.dal.usermanager.schema.UserManagerSchema
import ru.yandex.crm.apphost.kotlin.handlers.teammanager.repository.OrganizationTeamRepository
import ru.yandex.crm.apphost.kotlin.handlers.teammanager.service.OrganizationTeamService
import ru.yandex.crm.apphost.kotlin.handlers.teammanager.service.impl.OrganizationTeamServiceImpl
import ru.yandex.crm.apphost.kotlin.handlers.teammanager.service.mappers.OrganizationTeamMapper
import ru.yandex.crm.apphost.kotlin.handlers.teammanager.service.mappers.impl.OrganizationTeamMapperImpl
import ru.yandex.crm.apphost.kotlin.handlers.teammanager.ut.repository.MockOrganizationTeamRepository
import ru.yandex.crm.library.kotlin.database.hibernate.*
import ru.yandex.crm.proto.gallifrey.teammanager.Teammanager.TeamData
import ru.yandex.crm.proto.gallifrey.teammanager.Teammanager.TeamName
import ru.yandex.crm.proto.gallifrey.teammanager.Teammanager.UpdateTeamData
import java.util.UUID
import javax.sql.DataSource
import kotlin.test.assertEquals
import kotlin.test.assertNotNull
import kotlin.test.assertTrue

private val hibernateConfig = HibernateConfig().apply {
    dialect = "org.hibernate.dialect.H2Dialect"
    showSql = true
    hbm2ddlAuto = "create"
}

private val dataSource = BasicDataSource().apply {
    driverClassName = "org.h2.Driver"
    url = "jdbc:h2:mem:myDb;DB_CLOSE_DELAY=-1"
}

private val serviceModule = module {
    single { hibernateConfig } bind HibernateConfig::class
    single { dataSource } bind DataSource::class
    single { HibernateSessionManagerImpl(get(), get(), getAll()) } bind HibernateSessionManager::class
    single { OrganizationManagerSchema() } bind HibernateSchema::class
    single { UserManagerSchema() } bind HibernateSchema::class
    factory { params -> MockOrganizationTeamRepository() } bind OrganizationTeamRepository::class
    single<OrganizationTeamMapper> { OrganizationTeamMapperImpl() }

    single { OrganizationTeamServiceImpl(get()) } bind OrganizationTeamService::class
}

class OrganizationTeamServiceTests: KoinTest {
    @BeforeEach
    fun beforeEach() {
        startKoin {
            modules(serviceModule)
        }
    }

    @AfterEach
    fun afterEach() {
        stopKoin()
    }

    @Test
    fun `Should return all available teams for organization`() {
        val service by inject<OrganizationTeamService>()

        val teams = service.getAllOrganizationTeams(1)

        assertTrue(teams.size == 3)
    }

    @Test
    fun `Should return team for specified id`() {
        val service by inject<OrganizationTeamService>()

        val teamId = UUID.fromString("f1f8ff22-8f68-4111-8aea-c40f515c0948")
        val team = service.getOrganizationTeamById(1, teamId)

        assertNotNull(team)
        assertEquals(teamId.toString(), team.id)
    }

    @Test
    fun `Should throw exception when get non-existing team`() {
        val service by inject<OrganizationTeamService>()

        val teamId = UUID(0L, 0L)
        assertThrows<Exception> { service.getOrganizationTeamById(1, teamId) }
    }

    @Test
    fun `Should throw exception when get deleted team`() {
        val deletedTeam = transaction {
            val repository by inject<OrganizationTeamRepository>()
            repository.findAll().first { it.isDeleted }
        }
        val service by inject<OrganizationTeamService>()

        assertThrows<Exception> { service.getOrganizationTeamById(1, deletedTeam.id!!) }
    }

    @Test
    fun `Should create new team`() {
        val service by inject<OrganizationTeamService>()

        val randomSecurityProfileId = UUID.randomUUID()
        val randomTeamName = "Random team ${UUID.randomUUID().toString()}"
        val teamData = TeamData.newBuilder()
            .setOrganizationId(2)
            .setSecurityProfileId(randomSecurityProfileId.toString())
            .addAllNames(
                mutableListOf( TeamName.newBuilder()
                .setName(randomTeamName)
                .setLanguageCode("en")
                .build()))
            .build()
        val newTeam = service.createOrganizationTeam(teamData)

        assertNotNull(newTeam)
        assertEquals(randomSecurityProfileId.toString(), newTeam.data.securityProfileId)
        assertEquals(2L, newTeam.data.organizationId)
        assertEquals(randomTeamName, newTeam.data.namesList.first().name)
    }

    @Test
    fun `Should update existing team`() {
        val existingTeam = transaction {
            val repository by inject<OrganizationTeamRepository>()
            repository.findAll().first { !it.isDeleted }
        }

        val namesData = existingTeam.names.map {
            TeamName.newBuilder()
                .setName("${it.name} Updated")
                .setLanguageCode(it.id!!.languageCode)
                .build()
        }.toMutableList()
        val existingNamesCount = namesData.size
        namesData.add(
            TeamName.newBuilder()
                .setName("New Name")
                .setLanguageCode("tt")
                .build()
        )
        val updateData = UpdateTeamData.newBuilder()
            .addAllNewNames(namesData)
            .build()

        val service by inject<OrganizationTeamService>()
        val updatedTeamData =
            service.updateOrganizationTeam(updateData, existingTeam.organizationId, existingTeam.id!!)

        assertNotNull(updatedTeamData)
        assertEquals(existingNamesCount + 1, updatedTeamData.data.namesList.size)
    }

    @Test
    fun `Should throw exception when update non-existing team`() {
        val updateData = UpdateTeamData.newBuilder()
            .addAllNewNames(
                mutableListOf(
                    TeamName.newBuilder()
                        .setName("New Name")
                        .setLanguageCode("tt")
                        .build()
                )
            )
            .build()

        val service by inject<OrganizationTeamService>()
        assertThrows<Exception> { service.updateOrganizationTeam(updateData, 100, UUID(0L, 0L)) }
    }

    @Test
    fun `Should delete successfully`() {
        val repository by inject<OrganizationTeamRepository>()
        val existingTeam = transaction {
            repository.findAll().first { !!it.isDeleted }
        }
        val service by inject<OrganizationTeamService>()

        service.deleteOrganizationTeam(existingTeam.organizationId!!, existingTeam.id!!)

        val deletedTeam = transaction {
            repository.findOne(existingTeam.id!!)
        }

        assertNotNull(deletedTeam)
        assertTrue(deletedTeam.isDeleted)
    }
}
