package ru.yandex.crm.apphost.kotlin.handlers.usermanager.service.impl

import ru.yandex.crm.apphost.kotlin.dal.usermanager.User
import ru.yandex.crm.apphost.kotlin.handlers.usermanager.repository.OrganizationUserRepository
import ru.yandex.crm.apphost.kotlin.handlers.usermanager.repository.UserRepository
import ru.yandex.crm.apphost.kotlin.handlers.usermanager.service.UserService
import ru.yandex.crm.apphost.kotlin.handlers.usermanager.service.mappers.UserEntityMapper
import ru.yandex.crm.apphost.kotlin.handlers.usermanager.util.AuditUtil
import ru.yandex.crm.apphost.proto.auditor.Auditor.UpdateEntityAttributeLog
import ru.yandex.crm.library.kotlin.database.hibernate.getRepository
import ru.yandex.crm.library.kotlin.database.hibernate.transaction
import ru.yandex.crm.proto.gallifrey.usermanager.Usermanager
import ru.yandex.crm.proto.gallifrey.usermanager.Usermanager.UpdateUserData
import ru.yandex.crm.proto.gallifrey.usermanager.Usermanager.UserData

class UserServiceImpl(
    private val userMapper: UserEntityMapper,
) : UserService {

    override fun getAllUsers(): List<Usermanager.User> {
        val users = transaction {
            val repository = getRepository<UserRepository>()
            repository.findAll()
        }
        return users.map { userMapper.toProtobufModel(it) }
    }

    override fun getOrganizationUsers(organizationId: Long, pattern: String?): List<Usermanager.User> {
        val organizationUsers = transaction {
            val repository = getRepository<OrganizationUserRepository>()
            if (pattern == null) {
                repository.findAllOrganizationUsers(organizationId)
            }else{
                repository.findOrganizationUsersByPattern(organizationId,pattern)
            }
        }
        return organizationUsers.map { userMapper.toProtobufModel(it.user) }
    }

    override fun getUser(userId: Long): Usermanager.User {
        val user = transaction {
            val repository = getRepository<UserRepository>()
            repository.findOne(userId)
        } ?: error("User with id: $userId not found")
        return userMapper.toProtobufModel(user)
    }

    override fun createUser(user: UserData): Usermanager.User {
        val newUser = User(
            firstName = user.firstName,
            lastName = user.secondName,
        )
        val savedUser = transaction {
            val repository = getRepository<UserRepository>()
            repository.save(newUser)
        }
        return userMapper.toProtobufModel(savedUser)
    }

    override fun updateUser(userId: Long, user: UpdateUserData): UserService.UpdateResult {
        val logs = mutableListOf<UpdateEntityAttributeLog>()
        val savedUser = transaction {
            val repository = getRepository<UserRepository>()
            val entity = repository.findOne(userId)
                ?: error("User with id: $userId not found")

            if (user.hasNewFirstName()) {
                val log = AuditUtil.buildUpdateAttributeUserLog(
                    userId, "first_name", entity.firstName, user.newFirstName
                )
                logs.add(log)

                entity.firstName = user.newFirstName
            }
            if (user.hasNewSecondName()) {
                val log = AuditUtil.buildUpdateAttributeUserLog(
                    userId, "second_name", entity.lastName, user.newSecondName
                )
                logs.add(log)

                entity.lastName = user.newSecondName
            }

            repository.save(entity)
        }

        val updatedUser = userMapper.toProtobufModel(savedUser)
        return UserService.UpdateResult(updatedUser, logs)
    }

    override fun archiveUser(userId: Long): Usermanager.User {
        TODO("Not yet implemented")
    }
}
