package ru.yandex.crm.apphost.kotlin.handlers.usermanager.ut.service

import org.apache.commons.dbcp2.BasicDataSource
import org.junit.jupiter.api.AfterEach
import org.junit.jupiter.api.BeforeEach
import org.junit.jupiter.api.Test
import org.junit.jupiter.api.assertThrows
import org.koin.core.context.startKoin
import org.koin.core.context.stopKoin
import org.koin.dsl.bind
import org.koin.dsl.module
import org.koin.test.KoinTest
import org.koin.test.inject
import ru.yandex.crm.apphost.kotlin.dal.organizationmanager.schema.OrganizationManagerSchema
import ru.yandex.crm.apphost.kotlin.dal.usermanager.schema.UserManagerSchema
import ru.yandex.crm.apphost.kotlin.handlers.usermanager.repository.OrganizationUserRepository
import ru.yandex.crm.apphost.kotlin.handlers.usermanager.repository.UserRepository
import ru.yandex.crm.apphost.kotlin.handlers.usermanager.service.UserService
import ru.yandex.crm.apphost.kotlin.handlers.usermanager.service.impl.UserServiceImpl
import ru.yandex.crm.apphost.kotlin.handlers.usermanager.service.mappers.UserEntityMapper
import ru.yandex.crm.apphost.kotlin.handlers.usermanager.service.mappers.impl.UserEntityMapperImpl
import ru.yandex.crm.apphost.kotlin.handlers.usermanager.ut.repository.MockOrganizationUserRepository
import ru.yandex.crm.apphost.kotlin.handlers.usermanager.ut.repository.MockUserRepository
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateConfig
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateSchema
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateSessionManager
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateSessionManagerImpl
import ru.yandex.crm.library.kotlin.test.isNotNull
import ru.yandex.crm.proto.gallifrey.usermanager.Usermanager.UpdateUserData
import ru.yandex.crm.proto.gallifrey.usermanager.Usermanager.UserData
import javax.sql.DataSource
import kotlin.test.assertEquals
import kotlin.test.assertNotNull

typealias ProtoUser = ru.yandex.crm.proto.gallifrey.usermanager.Usermanager.User

private val hibernateConfig = HibernateConfig().apply {
    dialect = "org.hibernate.dialect.H2Dialect"
    showSql = true
    hbm2ddlAuto = "create"
}

private val dataSource = BasicDataSource().apply {
    driverClassName = "org.h2.Driver"
    url = "jdbc:h2:mem:myDb;DB_CLOSE_DELAY=-1"
}

private val module = module {
    single { hibernateConfig } bind HibernateConfig::class
    single { dataSource } bind DataSource::class
    single { UserManagerSchema() } bind HibernateSchema::class
    single { OrganizationManagerSchema() } bind HibernateSchema::class
    single { HibernateSessionManagerImpl(get(), get(), getAll()) } bind HibernateSessionManager::class

    factory { params -> MockUserRepository(params.get()) } bind UserRepository::class
    factory { params -> MockOrganizationUserRepository(params.get()) } bind OrganizationUserRepository::class

    single { UserEntityMapperImpl() } bind UserEntityMapper::class

    factory { UserServiceImpl(get()) } bind UserService::class
}

class UserServiceTests : KoinTest {

    @BeforeEach
    fun beforeEach() {
        startKoin {
            modules(module)
        }
    }

    @AfterEach
    fun afterEach() {
        stopKoin()
    }

    @Test
    fun `get all users`() {
        val service by inject<UserService>()

        val users = service.getAllUsers()

        assert(users.size == 5)
        assert { users.all { it.userId.isNotNull() } }
        assert { users.all { it.data.firstName.isNotNull() } }
        assert { users.all { it.data.secondName.isNotNull() } }
    }

    @Test
    fun `get organization users`() {
        val service by inject<UserService>()

        val users1 = service.getOrganizationUsers(1)
        val users2 = service.getOrganizationUsers(2)

        assert(users1.isNotEmpty())
        assert(users2.isNotEmpty())
    }

    @Test
    fun `get organization users by pattern`() {
        val service by inject<UserService>()

        val users1 = service.getOrganizationUsers(1, "st1")

        assertEquals(1, users1.size)
    }

    @Test
    fun `get existing user`() {
        val service by inject<UserService>()

        val user1 = service.getUser(1)
        val user2 = service.getUser(2)

        assertNotNull(user1.userId)
        assertNotNull(user1.data.firstName)
        assertNotNull(user1.data.secondName)

        assertEquals(1L, user1.userId)
        assertEquals("first1", user1.data.firstName)
        assertEquals("last1", user1.data.secondName)

        assertNotNull(user2.userId)
        assertNotNull(user2.data.firstName)
        assertNotNull(user2.data.secondName)

        assertEquals(2L, user2.userId)
        assertEquals("first2", user2.data.firstName)
        assertEquals("last2", user2.data.secondName)
    }

    @Test
    fun `get not exiting user`() {
        val service by inject<UserService>()

        assertThrows<Exception> {
            service.getUser(-1)
        }
    }

    @Test
    fun `create user`() {
        val service by inject<UserService>()

        val user = UserData.newBuilder()
            .setFirstName("firstname1")
            .setSecondName("secondname1")
            .build()

        val savedUser = service.createUser(user)

        assertNotNull(savedUser.userId)
        assertEquals("firstname1", savedUser.data.firstName)
        assertEquals("secondname1", savedUser.data.secondName)
    }

    @Test
    fun `update user`() {
        val service by inject<UserService>()

        val id = 3L
        val user = UpdateUserData.newBuilder()
            .setNewFirstName("new firstname1")
            .setNewSecondName("new secondname1")
            .build()

        val updateResult = service.updateUser(id, user)
        val updatedUser = updateResult.updatedUser


        assertEquals("new firstname1", updatedUser.data.firstName)
        assertEquals("new secondname1", updatedUser.data.secondName)
    }

    private fun assert(action: () -> Boolean) {
        return assert(action.invoke())
    }
}
