package ru.yandex.crm.apphost.kotlin.handlers.userselector.service.impl

import ru.yandex.crm.apphost.kotlin.dal.organizationmanager.Organization
import ru.yandex.crm.apphost.kotlin.dal.usermanager.User
import ru.yandex.crm.apphost.kotlin.handlers.userselector.repository.OrganizationRepository
import ru.yandex.crm.apphost.kotlin.handlers.userselector.repository.UserRepository
import ru.yandex.crm.apphost.kotlin.handlers.userselector.service.AuthenticateService
import ru.yandex.crm.apphost.kotlin.handlers.userselector.service.models.AuthenticateResult
import ru.yandex.crm.apphost.proto.userselector.UserSelector.User.AuthenticationType
import ru.yandex.crm.apphost.proto.userselector.UserSelector.YandexService
import ru.yandex.crm.apphost.proto.userselector.UserSelector.YandexUser
import ru.yandex.crm.library.kotlin.database.hibernate.getRepository
import ru.yandex.crm.library.kotlin.database.hibernate.transaction

typealias ProtoUser = ru.yandex.crm.apphost.proto.userselector.UserSelector.User
typealias ProtoOrganization = ru.yandex.crm.apphost.proto.userselector.UserSelector.Organization

class AuthenticateServiceImpl : AuthenticateService {

    override fun processAuthenticate(
        yandexUser: YandexUser?,
        yandexService: YandexService?
    ): AuthenticateResult? {
        if (yandexUser != null) {
            return authenticateByYandexUser(yandexUser)
        }
        if (yandexService != null) {
            return authenticateByYandexService(yandexService)
        }

        return null
    }

    private fun authenticateByYandexUser(yandexUser: YandexUser): AuthenticateResult? {
        val uId = yandexUser.uId
        val (user, organizations) = transaction {
            val userRepository = getRepository<UserRepository>()
            val organizationRepository = getRepository<OrganizationRepository>()

            val entityUser = userRepository.getUserWithUID(uId) ?: return null
            val entityOrganizations = organizationRepository.getAllOrganizationsWithUserId(entityUser.id!!)
            val authenticatedUser = entityUser.toProtobufModel(AuthenticationType.YaStaffUser)
            val authenticatedOrganizations = entityOrganizations.map { it.toProtobufModel() }

            authenticatedUser to authenticatedOrganizations
        }
        return AuthenticateResult(user, organizations)
    }

    private fun authenticateByYandexService(service: YandexService): AuthenticateResult? {
        val tvmId = service.tvmId.toLong()
        val (user, organizations) = transaction {
            val userRepository = getRepository<UserRepository>()
            val organizationRepository = getRepository<OrganizationRepository>()

            val entityUser = userRepository.getUserWithTvmId(tvmId) ?: return null
            val entityOrganizations = organizationRepository.getAllOrganizationsWithUserId(entityUser.id!!)
            val authenticatedUser = entityUser.toProtobufModel(AuthenticationType.YaService)
            val authenticatedOrganizations = entityOrganizations.map { it.toProtobufModel() }

            authenticatedUser to authenticatedOrganizations
        }
        return AuthenticateResult(user, organizations)
    }

    private fun User.toProtobufModel(type: AuthenticationType): ProtoUser {
        return ProtoUser.newBuilder()
            .setUserId(id!!)
            .setAuthenticationType(type)
            .build()
    }

    private fun Organization.toProtobufModel(): ProtoOrganization {
        return ProtoOrganization.newBuilder()
            .setOrganizationId(id!!)
            .setPoolId(pool)
            .build()
    }
}
