package ru.yandex.crm.apphost.kotlin.handlers.userselector.ut.service

import org.apache.commons.dbcp2.BasicDataSource
import org.junit.jupiter.api.AfterEach
import org.junit.jupiter.api.BeforeEach
import org.junit.jupiter.api.Test
import org.koin.core.context.startKoin
import org.koin.core.context.stopKoin
import org.koin.dsl.bind
import org.koin.dsl.module
import org.koin.test.KoinTest
import org.koin.test.inject
import ru.yandex.crm.apphost.kotlin.dal.organizationmanager.schema.OrganizationManagerSchema
import ru.yandex.crm.apphost.kotlin.dal.usermanager.schema.UserManagerSchema
import ru.yandex.crm.apphost.kotlin.handlers.userselector.repository.OrganizationRepository
import ru.yandex.crm.apphost.kotlin.handlers.userselector.repository.UserRepository
import ru.yandex.crm.apphost.kotlin.handlers.userselector.service.AuthenticateService
import ru.yandex.crm.apphost.kotlin.handlers.userselector.service.impl.AuthenticateServiceImpl
import ru.yandex.crm.apphost.kotlin.handlers.userselector.ut.repository.MockOrganizationRepository
import ru.yandex.crm.apphost.kotlin.handlers.userselector.ut.repository.MockUserRepository
import ru.yandex.crm.apphost.proto.userselector.UserSelector.User.AuthenticationType
import ru.yandex.crm.apphost.proto.userselector.UserSelector.YandexService
import ru.yandex.crm.apphost.proto.userselector.UserSelector.YandexUser
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateConfig
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateSchema
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateSessionManager
import ru.yandex.crm.library.kotlin.database.hibernate.HibernateSessionManagerImpl
import javax.sql.DataSource
import kotlin.test.assertEquals
import kotlin.test.assertNotNull
import kotlin.test.assertNull

private val hibernateConfig = HibernateConfig().apply {
    dialect = "org.hibernate.dialect.H2Dialect"
    showSql = true
    hbm2ddlAuto = "none"
}

private val dataSource = BasicDataSource().apply {
    driverClassName = "org.h2.Driver"
    url = "jdbc:h2:mem:myDb;DB_CLOSE_DELAY=-1"
}

private val repositoryModule = module {
    single { hibernateConfig } bind HibernateConfig::class
    single { dataSource } bind DataSource::class
    single { UserManagerSchema() } bind HibernateSchema::class
    single { OrganizationManagerSchema() } bind HibernateSchema::class
    single { HibernateSessionManagerImpl(get(), get(), getAll()) } bind HibernateSessionManager::class

    factory { MockOrganizationRepository() } bind OrganizationRepository::class
    factory { MockUserRepository() } bind UserRepository::class
}

private val serviceModule = module {
    single { AuthenticateServiceImpl() } bind AuthenticateService::class
}

class AuthenticateServiceTests : KoinTest {

    @BeforeEach
    fun beforeEach() {
        startKoin {
            modules(serviceModule, repositoryModule)
        }
    }

    @AfterEach
    fun afterEach() {
        stopKoin()
    }

    @Test
    fun `authenticate with null yandex user and null yandex service`() {
        val service: AuthenticateService by inject()

        val yandexUser: YandexUser? = null
        val yandexService: YandexService? = null

        val actualResult = service.processAuthenticate(yandexUser, yandexService)

        assertNull(actualResult)
    }

    @Test
    fun `authenticate with yandex user and null yandex service`() {
        val service: AuthenticateService by inject()

        val yandexUser = YandexUser.newBuilder()
            .setUId(1)
            .build()
        val yandexService: YandexService? = null

        val actualResult = service.processAuthenticate(yandexUser, yandexService)

        assertNotNull(actualResult)

        val user = actualResult.user
        val organizations = actualResult.organizations

        assertNotNull(user)
        assertNotNull(organizations)
        assertEquals(1L, user.userId)
        assertEquals(AuthenticationType.YaStaffUser, user.authenticationType)
        assert(organizations.size == 1)
    }

    @Test
    fun `authenticate with null yandex user and yandex service`() {
        val service: AuthenticateService by inject()

        val yandexUser: YandexUser? = null
        val yandexService = YandexService.newBuilder()
            .setTvmId(1)
            .build()

        val actualResult = service.processAuthenticate(yandexUser, yandexService)

        assertNotNull(actualResult)

        val user = actualResult.user
        val organizations = actualResult.organizations

        assertNotNull(user)
        assertNotNull(organizations)
        assertEquals(1L, user.userId)
        assertEquals(AuthenticationType.YaService, user.authenticationType)
        assert(organizations.size == 1)
    }

    @Test
    fun `authenticate with yandex user and yandex service`() {
        val service: AuthenticateService by inject()

        val yandexUser = YandexUser.newBuilder()
            .setUId(3)
            .build()
        val yandexService = YandexService.newBuilder()
            .setTvmId(2)
            .build()

        val actualResult = service.processAuthenticate(yandexUser, yandexService)

        assertNotNull(actualResult)

        val user = actualResult.user
        val organizations = actualResult.organizations

        assertNotNull(user)
        assertNotNull(organizations)
        assertEquals(3, user.userId)
        assertEquals(AuthenticationType.YaStaffUser, user.authenticationType)
        assert(organizations.size == 1)
    }

    @Test
    fun `authenticate with wrong yandex user uid`() {
        val service: AuthenticateService by inject()

        val yandexUser = YandexUser.newBuilder()
            .setUId(-1)
            .build()
        val yandexService: YandexService? = null

        val actualResult = service.processAuthenticate(yandexUser, yandexService)

        assertNull(actualResult)
    }

    @Test
    fun `authenticate with wrong yandex service tvm id`() {
        val service: AuthenticateService by inject()

        val yandexUser: YandexUser? = null
        val yandexService: YandexService = YandexService.newBuilder()
            .setTvmId(-1)
            .build()

        val actualResult = service.processAuthenticate(yandexUser, yandexService)

        assertNull(actualResult)
    }
}
